#' printRmInvalid
#
#' @description printRmInvalid prints to the console the lines that were registered to be invalid in the raw ESM dataset.
#
#' @param rmInvalidList a list. Pass the result obtained from function \code{rmInvalid} as argument.
#
#' @param smr a character string. Specify the level of detail you want to be displayed in the console. There are 3 options. The default option is 'tabulate', which displays a table, indicating the number of removed questionnaires for each ESM version. The next option is 'detail', which displays the entire removed questionnaires. The last option is 'both', which displays both the table and the detailed information.
#
#' @param RELEVANTVN_ES a list. This list is generated by function \code{\link{setES}} and it is extended once either by function \code{\link{genDateTime}} or by function \code{\link{splitDateTime}}.
#
#' @return In addition to the information the user wants to be displayed (see argument \code{smr}), the function returns a list containing the KEY value (see function \code{\link{genKey}}) of all the questionnaires that have been removed from each of the raw ESM datasets.
#
#' @importFrom stats addmargins
#
#' @examples
#' # o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o
#' # Prerequisites in order to execute printRmInvalid. Start -----------
#' # Use example list delivered with the package
#' RELEVANTVN_ES <- RELEVANTVN_ESext
#' # keyLsNew is a list of datasets, also delivered with the package
#' rmInvLs <- rmInvalid(keyLsNew, RELEVANTVN_ES)
#' # Prerequisites in order to execute printRmInvalid. End -------------
#' # ------------------------------------------------------
#' # Run function 10 of 29; see esmprep functions' hierarchy.
#' # ------------------------------------------------------
#' # rmInvLs is the result of function 'rmInvalid'. Display its result
#' # in the console both tablulated and in detail.
#' key_rmLs <- printRmInvalid(rmInvLs, smr="both", RELEVANTVN_ES)
#' # Display the list containing the KEY values of all questionnaires
#' # that have been removed.
#' key_rmLs
#' # Since there have been warning messages in 4 of the 6 datasets,
#' # the first ESM item (name: V1) was automatically converted to class
#' # character, although it is numeric. So we'll re-convert V1's class.
#' # Check V1 prior to conversion
#' str(rmInvLs[["dfValid"]][[2]][1:2])
#' rmInvLs[["dfValid"]] <- sapply(rmInvLs[["dfValid"]], function(x) {
#'      x[,"V1"] <- as.numeric(x[,"V1"])
#'      return(x) } )
#' # Check V1 after conversion
#' str(rmInvLs[["dfValid"]][[2]][1:2])
#' # o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o=o
#
#' @seealso Exemplary code (fully executable) in the documentation of \code{\link{esmprep}} (function 10 of 29).
#
#' @export
#
printRmInvalid <- function(rmInvalidList, RELEVANTVN_ES = NULL, smr = "tabulate") {

    if(class(rmInvalidList) != "list") {
        stop("Argument must be of class 'list'.")
    }

    if(!all(sapply(rmInvalidList[[1]], FUN = is.data.frame))) {
        stop("Element 1 of the list must be a list containing data frames only.")
    }

    if(length(rmInvalidList) != 4 | names(rmInvalidList)[3]!="rmInvalidFinished") {
        stop("Function 'printRmInvalid' must be provided with the result of function 'rmInvalid'. Therefore apply function 'rmInvalid' prior to this function.")
    }

    if(any(rmInvalidList[["noLinesRemovedAtAll"]])) {
        cat("!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!\n")
        cat("At least one line has been removed from a raw dataset due to being invalid, i.e. all values in the removed line(s) were missing; most likely this is due to a technical error.\n")
        cat("!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!-!\n\n")
    }
    
    
    # Error handling function for all set-up lists generated by setES and setREF.
    # Both lists RELEVANTVN_ES and RELEVANTVN_REF get extended either by function
    # genDateTime or by function splitDateTime!
    SETUPLISTCheck(RELEVANTINFO_ES=NULL,
    			   RELEVANTVN_ES=RELEVANTVN_ES,
    			   RELEVANTVN_REF=NULL)
    
    
    for(i in 1:length(rmInvalidList[["listInvalid"]])) {
        if(rmInvalidList[["noLinesRemovedAtAll"]][[i]]==FALSE) {
            next
        } else {
            # If there are any missing values in the variable that denotes the version
            # of the event sampling questionnaire they will be returned as NA. If any
            # entry in this variable consist only of empty characters a character value
            # of length 0 will be returned in variable 'svyName_i0'. Therefore the
            # variable 'svyName_i0' can contain 3 elements maximally.
            svyName_i0 <- gsub(" ", "", unique(rmInvalidList[["dfValid"]][[i]][,RELEVANTVN_ES[["ES_SVY_NAME"]]]))
            # Remove any possible NA or empty character.
            svyName_i <- svyName_i0[!is.na(svyName_i0) | !svyName_i0 == ""]
        }   
    }
    
    # Generate table with raw ESM datasets after invalid lines have
	# been removed.
	dfAfterRemovals <- addmargins(as.table(sapply(rmInvalidList[["dfValid"]], FUN = nrow)))
	
	# Generate table with raw ESM datasets only consisting of lines
	# that had been removed
	removals <- addmargins(
	as.table(sapply(rmInvalidList[["listInvalid"]], function(x) {
		ifelse(is.null(nrow(x)), 0, nrow(x)) })))
	
	# Compute number of lines before removals:
	beforeRemovals <- as.numeric(dfAfterRemovals) + as.numeric(removals)
	
	# Make output data.frame
	tblOut <- data.frame(versions=names(dfAfterRemovals),
	notRemoved=as.numeric(dfAfterRemovals), removed=as.numeric(removals), Sum=beforeRemovals)
	
	removedKey <- sapply(rmInvalidList[["listInvalid"]], function(x) x["KEY"])
    
    # Output to console
	if(smr == "detail") {
		
	    for(i in 1:length(rmInvalidList[["listInvalid"]])) {
	        if(rmInvalidList[["noLinesRemovedAtAll"]][[i]]==FALSE) {
	            next
	        } else {
	            cat("-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-\n")
	            cat(paste0("F i l t e r   n o . ", i, ": ", nrow(rmInvalidList[["listInvalid"]][[i]]), " line(s) have been removed from the raw dataset of questionnaire version '", svyName_i, "'.\n\n"))
	            cat("---------------------\n")
	            cat("Print removed line(s):\n")
	            cat("---------------------\n")
	            print(rmInvalidList[["listInvalid"]][[i]])
	        }
	        cat("-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-\n\n\n")
	    }
	    
	} else if(smr == "tabulate") {
		
		cat("-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-\n")
		cat("Summarizing table:\n")
		print(tblOut)
		cat("-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-\n\n\n")
		
	} else if(smr == "both") {
		
		for(i in 1:length(rmInvalidList[["listInvalid"]])) {
	        if(rmInvalidList[["noLinesRemovedAtAll"]][[i]]==FALSE) {
	            next
	        } else {
	            cat("-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-\n")
	            cat(paste0("F i l t e r   n o . ", i, ": ", nrow(rmInvalidList[["listInvalid"]][[i]]), " line(s) have been removed from the raw dataset of questionnaire version '", svyName_i, "'.\n\n"))
	            cat("---------------------\n")
	            cat("Print removed line(s):\n")
	            cat("---------------------\n")
	            print(rmInvalidList[["listInvalid"]][[i]])
	        }
	        cat("-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-\n\n\n")
	    }
	    
	    cat("-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-\n")
		cat("Summarizing table:\n")
		print(tblOut)
		cat("-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-o-\n\n\n")
		
	}
	# Return a named list. It contains as many elements as there are raw ESM
	# datasets. For each element the user can read the KEY identifier of the
	# line(s) that have been registered to be invalid and have therefore been
	# removed; in case the line is not invalid after all and shall be put back
	# into the dataset.
	removedKey
}
