% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/environment_name.r
\name{environment_name}
\alias{environment_name}
\alias{get_env_name}
\title{Retrieve the name of an environment}
\usage{
environment_name(
  env = parent.frame(),
  envir = NULL,
  envmap = NULL,
  matchname = FALSE,
  ignore = NULL,
  include_functions = FALSE
)
}
\arguments{
\item{env}{environment whose name is of interest.
It can be given as an object of class environment, as a string with the name of the environment,
or as a string with the memory address of the environment.
The latter is useful to find out if a given memory address is the reference of an environment object.
Note that the variable passed may or may \emph{not} exist in the calling environment, as the purpose
of this function is also to search for it (and return its name if it is an environment).
It defaults to parent.frame(), meaning that the name of the environment that calls this function is retrieved.}

\item{envir}{environment where \code{env} should be searched for. When \code{NULL}, \code{env} is searched in
the whole workspace, including packages and user-defined environments, recursively.}

\item{envmap}{data frame containing a lookup table with name-address pairs of environment names and
addresses to be used when searching for environment \code{env}. It defaults to \code{NULL} which means that the
lookup table is constructed on the fly with the environments defined in the \code{envir} environment
--if not \code{NULL}--, or in the whole workspace if \code{envir=NULL}.
See the details section for more information on its structure.}

\item{matchname}{flag indicating whether the match for \code{env} is based on its name or on its
memory address. In the latter case all environments sharing the same memory address of the given
environment are returned. Such scenario happens when, for instance, different
environment objects have been defined equal to another environment (as in \code{env1 <- env}).
It defaults to \code{FALSE}.}

\item{ignore}{one or more environment names to ignore if found during the search. These environments
are removed from the output. It should be given as a character array if more than one environments
should be ignored. See the details section for more information.}

\item{include_functions}{flag indicating whether to look for user-defined environments inside function
execution environments. This should be used with care because in a complicated function chain, some function
execution environments may contain environments that point to other environments (e.g. the 'envclos' environment
in the \code{eval()} function when running tests using the \code{test_that} package).}
}
\value{
If \code{matchname=FALSE} (the default), an array containing the names of all the environments
(defined in the \code{envir} environment if \code{envir} is not \code{NULL}) having the same memory address
as the \code{env} environment.

If \code{matchname=TRUE}, the environment name contained in \code{env} is used in addition to the memory
address to check the matched environments (potentially many if they have the same memory address)
so that only the environments having the same name and address as the \code{env} environment are returned.
Note that several environments may be found if environments with the same name are defined in
different environments.
WARNING: in this case, the name is matched exactly as the expression given in \code{env}. So for instance,
if \code{env=globalenv()$env1} the name \code{"globalenv()$env1"} is checked and this will not return any
environments since no environment can be called like that. For such scenario call the function with
parameter \code{env=env1} instead, or optionally with \code{env=env1} and \code{envir=globalenv()}
if the \code{env1} environment should be searched for just in the global environment.

If \code{env} is not found or it is not an environment, \code{NULL} is returned.
}
\description{
Retrieve the name of an environment as the \code{\link{environmentName}} function of the base package does,
but extending its functionality to retrieving the names of user-defined environments and function
execution environments.
}
\details{
If \code{env} is an environment it is searched for in the \code{envir} environment using its memory address.
If \code{env} is a string containing a valid 16-digit memory address (enclosed in < >), the memory address
itself is searched for among the defined environments in the \code{envir} environment.
In both cases, if \code{envir=NULL} the search is carried out in the whole workspace. 
 
It may happen that more than one environment exist with the same memory address (for instance
if an environment is a copy of another environment). In such case, if \code{matchname=FALSE},
the names of ALL the environments matching \code{env}'s memory address are returned. Otherwise,
only the environments matching the given name are returned.

If \code{envmap} is passed it should be a data frame providing an address-name pair lookup table
of environments and should contain at least the following columns:
\itemize{
\item{\code{location}} for user-defined environments, the name of the environment where the environment
is located; otherwise \code{NA}.
\item{\code{pathname}} the full \emph{environment path} to reach the environment separated by \code{$}
(e.g. \code{"env1$env$envx"})
\item{\code{address}} an 8-digit (32-bit architectures) thru 16-digit (64-bit architectures) memory address
of the environment given in \code{pathname} enclosed in < > (e.g. \code{"<0000000007DCFB38>"}
(64-bit architectures))
Be ware that Linux Debian distributions may have a 12-digit memory address representation.
So the best way to know is to check a memory address by calling e.g. `address("x")`.
}
Passing an \code{envmap} lookup table is useful for speedup purposes, in case several calls to this
function will be performed in the context of an unchanged set of defined environments.
Such \code{envmap} data frame can be created by calling \link{get_env_names}.
Use this parameter with care, as the matrix passed may not correspond to the actual mapping of existing
environments to their addresses and in that case results may be different from those expected.

The following example illustrates the use of the \code{ignore} parameter:

\code{for (e in c(globalenv(), baseenv())) { print(environment_name(e, ignore="e")) }}

That is, we iterate on a set of environments and we don't want the loop variable (an environment itself)
to show up as part of the output generated by the call to \code{environment_name()}.
}
\examples{
# Retrieve name of a user-defined environment
env1 <- new.env()
environment_name(env1)                   		# "env1"

# Retrieve the name of an environment given as a memory address
env1_address = get_obj_address(globalenv()$env1)
environment_name(env1_address)           		# "env1"

# Create a copy of the above environment
env1_copy <- env1
environment_name(env1)                   		# "env1" "env1_copy"
# Retrieve just the env1 environment name
environment_name(env1, matchname=TRUE)   		# "env1"

# Retrieve the name of an environment defined within another environment
with(env1, envx <- new.env())
environment_name(env1$envx)              		# "env1$envx" "env1_copy$envx"
environment_name(env1$envx, matchname=TRUE)
  ## NULL, because the environment name is "envx", NOT "env1$envx"

# Get a function's execution environment name
with(env1, f <- function() { cat("We are inside function", environment_name()) })  
    ## "We are inside function env1$f"
}
