#' @title Diversity indices
#'
#' @description Computes and prints all the diversity indices
#'
#' @param x Vector of dimension S (spicies) with the numbers of observed individuals in each spicy. NA values are allowed. 0 values are converted to NA.
#' @param groups Vector of dimension S of factors indicating the groups.
#'
#' @return{No return value. It prints the value of all indicators}
#'
#' @seealso \link{shannon}, \link{dec_shannon}, \link{dominance}, \link{equitability}, \link{evenness}, \link{margalev}, \link{menhinick}, \link{simpson_D}, \link{simpson_E}
#'
#' @references
#' "Arnaud Barat, Andreu Sansó, Maite Arilla-Osuna, Ruth Blasco, Iñaki Pérez-Fernández, Gabriel Cifuentes-Alcobenda, Rubén Llorente, Daniel Vivar-Ríos, Ella Assaf, Ran Barkai, Avi Gopher, & Jordi Rosell-Ardèvol (2025): Quantifying Diversity through Entropy Decomposition. Insights into Hominin Occupation and Carcass Processing  at Qesem cave"
#'
#' @examples
#' data(Qesem_s)
#' all_indices(Qesem_s$HU)
#' all_indices(Qesem_s$HU, Qesem_s$Group)
#'
#' @export
all_indices <- function(x, groups = NULL){
  if (!is.null(groups)) {
    sh <- dec_shannon(x, groups)
    cat("Shannon's diversity (Total):", sh$shannon,"\n")
    cat("Decomposing Shannon's diversity:\n")
    cat("    Within groups: ", sh$within,"\n")
    cat("    Between groups:", sh$between,"\n")
    cat("    Fraction Between/Total:", sh$between/sh$shannon,"\n")
    cat("  Entropy within each group\n")
    print(sh$g)
    cat("\n")
  } else {cat("Shannon's diversity:", shannon(x),"\n")}
  cat("Equitability (J Pielou):", equitability(x),"\n")
  cat("Simpson's dominance:", simpson_D(x),"\n")
  cat("Dominance:", dominance(x),"\n")
  cat("Simpson's evenness:", simpson_E(x),"\n")
  cat("Evenness:", evenness(x),"\n")
  cat("Menhinick's Index:", menhinick(x),"\n")
  cat("Margalev's Index:", margalev(x),"\n\n")
}
