% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/momentum.R
\name{momentum}
\alias{momentum}
\title{Contribution of Individual Variables to Time Series Dissimilarity}
\usage{
momentum(
  tsl = NULL,
  distance = "euclidean",
  diagonal = TRUE,
  bandwidth = 1,
  lock_step = FALSE,
  robust = TRUE
)
}
\arguments{
\item{tsl}{(required, time series list) list of zoo time series. Default: NULL}

\item{distance}{(optional, character vector) name or abbreviation of the distance method. Valid values are in the columns "names" and "abbreviation" of the dataset \link{distances}. Default: "euclidean".}

\item{diagonal}{(optional, logical vector). If TRUE, diagonals are included in the dynamic time warping computation. Default: TRUE}

\item{bandwidth}{(optional, numeric) Proportion of space at each side of the cost matrix diagonal (aka \emph{Sakoe-Chiba band}) defining a valid region for dynamic time warping, used to control the flexibility of the warping path. This method prevents degenerate alignments due to differences in magnitude between time series when the data is not properly scaled. If \code{1} (default), DTW is unconstrained. If \code{0}, DTW is fully constrained and the warping path follows the matrix diagonal. Recommended values may vary depending on the nature of the data. Ignored if \code{lock_step = TRUE}. Default: 1.}

\item{lock_step}{(optional, logical vector) If TRUE, time series captured at the same times are compared sample wise (with no dynamic time warping). Requires time series in argument \code{tsl} to be fully aligned, or it will return an error. Default: FALSE.}

\item{robust}{(required, logical). If TRUE (default), importance scores are computed using the least cost path of the complete time series as reference. Setting it to FALSE allows to replicate importance scores of the previous versions of this package. This option is irrelevant when \code{lock_step = TRUE}. Default: TRUE}
}
\value{
data frame:
\itemize{
\item \code{x}: name of the time series \code{x}.
\item \code{y}: name of the time series \code{y}.
\item \code{psi}: psi score of \code{x} and \code{y}.
\item \code{variable}: name of the individual variable.
\item \code{importance}: importance score of the variable.
\item \code{effect}: interpretation of the "importance" column, with the values "increases similarity" and "decreases similarity".
\item \code{psi_only_with}: psi score of the variable.
\item \code{psi_without}: psi score without the variable.
\item \code{psi_difference}: difference between \code{psi_only_with} and \code{psi_without}.
\item \code{distance}: name of the distance metric.
\item \code{diagonal}: value of the argument \code{diagonal}.
\item \code{lock_step}: value of the argument \code{lock_step}.
\item \code{robust}: value of the argument \code{robust}.
}
}
\description{
This function measures the contribution of individual variables to the dissimilarity between pairs of time series to help answer the question \emph{what makes two time series more or less similar?}

Three key values are required to assess individual variable contributions:
\itemize{
\item \strong{psi}: dissimilarity when all variables are considered.
\item \strong{psi_only_with}: dissimilarity when using only the target variable.
\item \strong{psi_without}: dissimilarity when removing the target variable.
}

The values \code{psi_only_with} and \code{psi_without} can be computed in two different ways defined by the argument \code{robust}.
\itemize{
\item \code{robust = FALSE}: This method replicates the importance algorithm released with the first version of the package, and it is only recommended when the goal to compare new results with previous studies. It normalizes \code{psi_only_with} and \code{psi_without} using the least cost path obtained from the individual variable. As different variables may have different least cost paths for the same time series, normalization values may change from variable to variable, making individual importance scores harder to compare.
\item \code{robust = TRUE} (default, recommended): This a novel version of the importance algorithm that yields more stable and comparable solutions. It uses the least cost path of the complete time series to normalize \code{psi_only_with} and \code{psi_without}, making importance scores of separate variables fully comparable.
}

The individual importance score of each variable (column "importance" in the output data frame) is based on different expressions depending on the \code{robust} argument, even when \code{lock_step = TRUE}:
\itemize{
\item \code{robust = FALSE}: Importance is computed as \code{((psi - psi_without) * 100)/psi} and interpreted as "change in similarity when a variable is removed".
\item \code{robust = TRUE}: Importance is computed as \code{((psi_only_with - psi_without) * 100)/psi} and interpreted as "relative dissimilarity induced by the variable expressed as a percentage".
}

In either case, positive values indicate that the variable contributes to dissimilarity, while negative values indicate a net contribution to similarity.

This function allows computing dissimilarity between pairs of time series using different combinations of arguments at once. For example, when the argument \code{distance} is set to \code{c("euclidean", "manhattan")}, the output data frame will show two dissimilarity scores for each pair of time series, one based on euclidean distances, and another based on manhattan distances. The same happens for most other parameters.

This function supports a parallelization setup via \code{\link[future:plan]{future::plan()}}, and progress bars provided by the package \href{https://CRAN.R-project.org/package=progressr}{progressr}.
}
\examples{

#progress bar
# progressr::handlers(global = TRUE)

tsl <- tsl_initialize(
  x = distantia::albatross,
  name_column = "name",
  time_column = "time"
) |>
  tsl_transform(
    f = f_scale_global
  )

df <- momentum(
  tsl = tsl,
  lock_step = TRUE #to speed-up example
  )

#focus on important columns
df[, c(
  "x",
  "y",
  "variable",
  "importance",
  "effect"
  )]

}
\seealso{
Other momentum: 
\code{\link{momentum_dtw}()},
\code{\link{momentum_ls}()}
}
\concept{momentum}
