% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/distantia_model_frame.R
\name{distantia_model_frame}
\alias{distantia_model_frame}
\title{Dissimilarity Model Frame}
\usage{
distantia_model_frame(
  response_df = NULL,
  predictors_df = NULL,
  composite_predictors = NULL,
  scale = TRUE,
  distance = "euclidean"
)
}
\arguments{
\item{response_df}{(required, data frame) output of \code{\link[=distantia]{distantia()}}, \code{\link[=distantia_ls]{distantia_ls()}}, \code{\link[=distantia_dtw]{distantia_dtw()}}, or \code{\link[=distantia_time_delay]{distantia_time_delay()}}. Default: NULL}

\item{predictors_df}{(required, data frame or sf data frame) data frame with numeric predictors for the the model frame. Must have a column with the time series names in \code{response_df$x} and \code{response_df$y}. If \code{sf} data frame, the column "geographic_distance" with distances between pairs of time series is added to the model frame. Default: NULL}

\item{composite_predictors}{(optional, list) list defining composite predictors. For example, \code{composite_predictors = list(a = c("b", "c"))} uses the columns \code{"b"} and \code{"c"} from \code{predictors_df} to generate the predictor \code{a} as the multivariate distance between \code{"b"} and \code{"c"} for each pair of time series in \code{response_df}. Default: NULL}

\item{scale}{(optional, logical) if TRUE, all predictors are scaled and centered with \code{\link[=scale]{scale()}}. Default: TRUE}

\item{distance}{(optional, string) Method to compute the distance between predictor values for all pairs of time series in \code{response_df}. Default: "euclidean".}
}
\value{
data frame: with attributes "predictors", "response", and "formula".
}
\description{
This function generates a model frame for statistical or machine learning analysis from these objects:

\itemize{
\item: Dissimilarity data frame generated by \code{\link[=distantia]{distantia()}}, \code{\link[=distantia_ls]{distantia_ls()}}, \code{\link[=distantia_dtw]{distantia_dtw()}}, or \code{\link[=distantia_time_delay]{distantia_time_delay()}}. The output model frame will have as many rows as this data frame.
\item: Data frame with static descriptors of the time series. These descriptors are converted to distances between pairs of time series via \code{\link[=distance_matrix]{distance_matrix()}}.
\item: List defining composite predictors. This feature allows grouping together predictors that have a common meaning. For example, \verb{composite_predictors = list(temperature = c("temperature_mean", "temperature_min", "temperature_max")} generates a new predictor named "temperature", which results from computing the multivariate distances between the vectors of temperature variables of each pair of time series. Predictors in one of such groups will be scaled before distance computation if their maximum standard deviations differ by a factor of 10 or more.
}

The resulting data frame contains the following columns:
\itemize{
\item \code{x} and \code{y}: names of the pair of time series represented in the row.
\item response columns in \code{response_df}.
\item predictors columns: representing the distance between the values of the given static predictor between \code{x} and \code{y}.
\item (optional) \code{geographic_distance}: If \code{predictors_df} is an sf \code{sf} data frame, then geographic distances are computed via \code{\link[sf:geos_measures]{sf::st_distance()}}.
}

This function supports a parallelization setup via \code{\link[future:plan]{future::plan()}}.
}
\examples{

#covid prevalence in California counties
tsl <- tsl_initialize(
  x = covid_prevalence,
  name_column = "name",
  time_column = "time"
) |>
  #subset to shorten example runtime
  tsl_subset(
    names = 1:5
  )

#dissimilarity analysis
df <- distantia_ls(tsl = tsl)

#combine several predictors
#into a new one
composite_predictors <- list(
  economy = c(
    "poverty_percentage",
    "median_income",
    "domestic_product"
    )
)

#generate model frame
model_frame <- distantia_model_frame(
  response_df = df,
  predictors_df = covid_counties,
  composite_predictors = composite_predictors,
  scale = TRUE
)

head(model_frame)

#names of response and predictors
#and an additive formula
#are stored as attributes
attributes(model_frame)$predictors

#if response_df is output of distantia():
attributes(model_frame)$response
attributes(model_frame)$formula

#example of linear model
# model <- lm(
#   formula = attributes(model_frame)$formula,
#   data = model_frame
# )
#
# summary(model)

}
\seealso{
Other distantia_support: 
\code{\link{distantia_aggregate}()},
\code{\link{distantia_boxplot}()},
\code{\link{distantia_cluster_hclust}()},
\code{\link{distantia_cluster_kmeans}()},
\code{\link{distantia_matrix}()},
\code{\link{distantia_spatial}()},
\code{\link{distantia_stats}()},
\code{\link{distantia_time_delay}()},
\code{\link{utils_block_size}()},
\code{\link{utils_cluster_hclust_optimizer}()},
\code{\link{utils_cluster_kmeans_optimizer}()},
\code{\link{utils_cluster_silhouette}()}
}
\concept{distantia_support}
