% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_densityMaps_overlay.R
\name{plot_densityMaps_overlay}
\alias{plot_densityMaps_overlay}
\title{Plot posterior probabilities of states/ranges on phylogeny from densityMaps}
\usage{
plot_densityMaps_overlay(
  densityMaps,
  colors_per_levels = NULL,
  add_ACE_pies = TRUE,
  cex_pies = 0.5,
  ace = NULL,
  ...,
  display_plot = TRUE,
  PDF_file_path = NULL
)
}
\arguments{
\item{densityMaps}{List of objects of class \code{"densityMap"}, typically generated with \code{\link[=prepare_trait_data]{prepare_trait_data()}},
that contains a phylogenetic tree and associated posterior probability of being in a given state/range along branches.
Each object (i.e., \code{densityMap}) corresponds to a state/range. If no color is provided for multi-area ranges, they will be interpolated.}

\item{colors_per_levels}{Named character string. To set the colors to use to map each state/range posterior probabilities. Names = states/ranges; values = colors.
If \code{NULL} (default), the color scale provided \code{densityMaps} will be used.}

\item{add_ACE_pies}{Logical. Whether to add pies of posterior probabilities of states/ranges at internal nodes on the mapped phylogeny. Default = \code{TRUE}.}

\item{cex_pies}{Numerical. To adjust the size of the ACE pies. Default = \code{0.5}.}

\item{ace}{Numerical matrix. To provide the posterior probabilities of ancestral states/ranges (characters) estimates (ACE) at internal nodes
used to plot the ACE pies. Rows are internal nodes. Columns are states/ranges. Values are posterior probabilities of each state per node.
Typically generated with \code{\link[=prepare_trait_data]{prepare_trait_data()}} in the \verb{$ace} slot.
If \code{NULL} (default), the ACE are extracted from the \code{densityMaps} with a possible slight discrepancy with the actual tip states
and estimated posterior probabilities of ancestral states.}

\item{...}{Additional arguments to pass down to \code{\link[phytools:plotSimmap]{phytools::plotSimmap()}} to control plotting.}

\item{display_plot}{Logical. Whether to display the plot generated in the R console. Default is \code{TRUE}.}

\item{PDF_file_path}{Character string. If provided, the plot will be saved in a PDF file following the path provided here. The path must end with ".pdf".}
}
\value{
If \code{display_plot = TRUE}, the function plots a time-calibrated phylogeny displaying the evolution of a categorical trait/biogeographic ranges.
If \code{PDF_file_path} is provided, the function exports the plot into a PDF file.
}
\description{
Plot on a time-calibrated phylogeny the evolution of a categorical trait/biogeographic ranges
summarized from \code{densityMaps} typically generated with \code{\link[=prepare_trait_data]{prepare_trait_data()}}.
Each branch is colored according to the posterior probability of being in a given state/range.
Color for each state/range are overlaid using transparency to produce a single plot for all states/ranges.
}
\examples{

# Load phylogeny and tip data
library(phytools)
data(eel.tree)
data(eel.data)

# Transform feeding mode data into a 3-level factor
eel_data <- stats::setNames(eel.data$feed_mode, rownames(eel.data))
eel_data <- as.character(eel_data)
eel_data[c(1, 5, 6, 7, 10, 11, 15, 16, 17, 24, 25, 28, 30, 51, 52, 53, 55, 58, 60)] <- "kiss"
eel_data <- stats::setNames(eel_data, rownames(eel.data))
table(eel_data)

# Manually define a Q_matrix for rate classes of state transition to use in the 'matrix' model
# Does not allow transitions from state 1 ("bite") to state 2 ("kiss") or state 3 ("suction")
# Does not allow transitions from state 3 ("suction") to state 1 ("bite")
# Set symmetrical rates between state 2 ("kiss") and state 3 ("suction")
Q_matrix = rbind(c(NA, 0, 0), c(1, NA, 2), c(0, 2, NA))

# Set colors per states
colors_per_levels <- c("limegreen", "orange", "dodgerblue")
names(colors_per_levels) <- c("bite", "kiss", "suction")

\donttest{ # (May take several minutes to run)
# Run evolutionary models to prepare trait data
eel_cat_3lvl_data <- prepare_trait_data(tip_data = eel_data, phylo = eel.tree,
    trait_data_type = "categorical",
    colors_per_levels = colors_per_levels,
    evolutionary_models = c("ER", "SYM", "ARD", "meristic", "matrix"),
    Q_matrix = Q_matrix,
    nb_simulations = 1000,
    plot_map = TRUE,
    plot_overlay = TRUE,
    return_best_model_fit = TRUE,
    return_model_selection_df = TRUE) }

# Load directly output
data(eel_cat_3lvl_data, package = "deepSTRAPP")

# Plot densityMaps one by one
plot(eel_cat_3lvl_data$densityMaps[[1]]) # densityMap for state n°1 ("bite")
plot(eel_cat_3lvl_data$densityMaps[[2]]) # densityMap for state n°1 ("kiss")
plot(eel_cat_3lvl_data$densityMaps[[3]]) # densityMap for state n°1 ("suction")

# Plot overlay of all densityMaps
plot_densityMaps_overlay(densityMaps = eel_cat_3lvl_data$densityMaps)

}
\seealso{
\code{\link[phytools:densityMap]{phytools::plot.densityMap()}} \code{\link[phytools:plotSimmap]{phytools::plotSimmap()}}
}
\author{
Maël Doré

Original functions by Liam Revell in R package \code{{phytools}}. Contact: \email{liam.revell@umb.edu}
}
