% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mds.ct.R
\name{mds.ct}
\alias{mds.ct}
\title{Continuous-time multidimensional scaling}
\usage{
mds.ct(
  d,
  argvals = NULL,
  weights = "equal",
  nbasis = 10,
  norder = 4,
  breaks = NULL,
  k = 2,
  smooth = "sandwich",
  lambda = exp(-10:20),
  gcvplot = TRUE,
  Im.tol = 1e-15,
  recenter = TRUE
)
}
\arguments{
\item{d}{matrix of dissimilarities}

\item{argvals}{time points}

\item{weights}{quadrature weights for the time points: either "equal" (default) 
or "trap" (trapezoidal, recommended for unequally spaced times)}

\item{nbasis}{number of B-spline basis functions}

\item{norder}{order of B-splines (degree plus 1); default is 4 (cubic splines)}

\item{breaks}{knots, including the endpoints of the time range}

\item{k}{number of principal coordinates}

\item{smooth}{bivariate smoothing method: "sandwich" (default) for the sandwich smoother,
"one-step" for a more traditional but slower tensor product smoother}

\item{lambda}{smoothing parameter}

\item{gcvplot}{logical: Should a plot of log lambda versus the GCV criterion be produced?}

\item{Im.tol}{tolerance for imaginary component of eigenvalues: imaginary components
with magnitude below this is set to zero, those above it generate an error.}

\item{recenter}{logical: Should the solution be double-centered?}
}
\value{
An object of class "mds.ct", which is a list with components
\item{funcs}{an object of class "\code{\link[fda]{fd}}" giving the \eqn{k} principal
coordinate functions}
\item{evals}{eigenvalues}
\item{argvals}{the given time points}
\item{GOF}{a \eqn{k \times 2} matrix giving the proportion of dissimilarity explained, 
according to the two definitions used by the \code{GOF} component of the output from \code{\link{cmdscale}}}
\item{sandwich}{output of \code{\link{s3}}, if \code{smooth=="sandwich"}}
\item{recenter}{value of the argument \code{recenter}}
}
\description{
A continuous-time version of classical multidimensional scaling.
}
\examples{

if (interactive() && 
    requireNamespace("fda", quietly = TRUE) &&
    requireNamespace("viridisLite", quietly = TRUE) &&
    requireNamespace("vegan", quietly = TRUE)) {
    
    require(ggplot2)
    
    data("handwrit", package = "fda")
    
    fives <- 5 * 0:280  + 1
    hw <- handwrit[ , 1, ]
    sd. <- .015
    
    hh <- cbind(hw[fives,], rnorm(281, sd=sd.))
    classical <- cmdscale(dist(hh), eig=TRUE)
    ctmds <- mds.ct(dist(hh), argvals=handwritTime[fives], nbasis=100, recenter=TRUE,
                    weights="trap", norder=7, lambda=exp(2:40))  
                    # a plot of GCV versus lambda is produced
    
    pro.classical <- vegan::procrustes(hw[fives,], classical, scale=FALSE)
    dat.classical <- as.data.frame(pro.classical$Yrot)
    pro.ctmds <- vegan::procrustes(hw[fives,], fda::eval.fd(ctmds$argvals, ctmds$funcs),
                                   scale=FALSE)
    dat.ctmds <- as.data.frame(matrix(pro.ctmds$translation, length(handwritTime), 2, byrow=TRUE) +
                                   fda::eval.fd(handwritTime, ctmds$funcs) \%*\% pro.ctmds$rotation)
    names(dat.classical) <- names(dat.ctmds) <- c("x", "y")
    dat.classical$time <- handwritTime[fives]
    dat.ctmds$time <- handwritTime
    
    # Plot of classical (dots) versus continuous-time (curve) MDS reconstructions
    # of the handwritten "fda" (grey), corrupted by noise
    g1 <- ggplot(hw, aes(x=X, y=Y)) + geom_point(color="darkgrey", size=.6) + coord_fixed() +
        geom_point(data = dat.classical, aes(x, y, color = time), size=1) +
        geom_point(data = dat.ctmds, aes(x, y, color = time), size=.6) +
        scale_color_gradientn(colors=viridisLite::plasma(50)) + 
        labs(x="", y="", color="Time (ms)", 
             title=bquote(sigma == .(sd.) * ", " 
                          ~ .(round(100*ctmds$GOF[2,1], 1)) * "\% explained")) 
    print(g1)
    
    g2 <- plot(ctmds)  # uses plot.mds.ct
    print(g2)
    
    g3 <- procrustes.plot.mds(ctmds, dat.classical[ , 1:2])
    print(g3)
    
}

}
\seealso{
\code{\link{cmdscale}}, for the classical version
}
\author{
Philip Tzvi Reiss <reiss@stat.haifa.ac.il>
}
