% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/credsubs.R
\name{credsubs}
\alias{credsubs}
\title{Constructs a credible subset pair}
\usage{
credsubs(
  params,
  design = NULL,
  FUN = function(x, params) {     params \%*\% t(x) },
  cred.level = 0.95,
  threshold = 0,
  method = c("asymptotic", "quantile"),
  step.down = TRUE,
  sides = c("both", "exclusive", "inclusive"),
  est.FUN = mean,
  var.FUN = sd,
  point.estimate = NULL,
  track = numeric(0),
  verbose = FALSE
)
}
\arguments{
\item{params}{A numeric matrix whose rows are draws from the posterior
distribution of either the regression surface or the
parameter vector.}

\item{design}{(Optional) A numeric matrix whose rows are covariate points
over which the band is to be constructed.}

\item{FUN}{(Optional) a function of the form \code{function(x, params)}
that takes a row of \code{design} and the entire \code{params}
matrix and returns a vector of the same length of \code{x}
representing the regression surface.}

\item{cred.level}{Numeric; the credible level.}

\item{threshold}{Numeric; the value of \code{FUN} above which
a covariate is included in the target subset.}

\item{method}{Either "asymptotic" (default) or "quantile"; see details.}

\item{step.down}{Logical (default \code{TRUE}); should the step-down
procedure be used?}

\item{sides}{One of "both" (default), "exclusive", or "inclusive".
Which bounds should be constructed?}

\item{est.FUN}{The function used to produce estimates of the regression
surface. Default is \code{mean}.}

\item{var.FUN}{The function used to quantify the variability of the
regression surface posterior. Default is \code{sd}.}

\item{point.estimate}{If not null, replaces the mean and sets the reference 
around which the standard error is computed.
Useful for bootstrapping methods.
Treated as a row of the \code{params} matrix.}

\item{track}{A numeric vector of indices indicating which rows (default none)
of the design matrix should have the sample of the corresponding
\code{FUN(x, params)} returned.}

\item{verbose}{Logical (default \code{FALSE}); print progress?}
}
\value{
An object of class \code{credsubs}, which contains:
\describe{
  \item{\code{exclusive}}{A logical vector indicating membership in
                          the exclusive credible subset.}
  \item{\code{inclusive}}{A logical vector indicating membership in
                          the inclusive credible subset.}
  \item{\code{cred.level}}{As provided.}
  \item{\code{threshold}}{As provided.}
  \item{\code{method}}{As provided.}
  \item{\code{step.down}}{As provided.}
  \item{\code{sides}}{As provided.}
  \item{\code{est}}{Posterior estimate of the regression surface.}
  \item{\code{est.FUN}}{As provided.}
  \item{\code{var}}{Summary of posterior variability of the regression
                    surface.}
  \item{\code{var.FUN}}{As provided.}
  \item{\code{W}}{An estimate of the extremal errors.}
  \item{\code{W.crit}}{The critical quantile of W.}
  \item{\code{trace}}{The posterior samples of the regression surface
                      indicated by the \code{track} argument.}
}
}
\description{
\code{credsubs} returns a credible subset pair over a finite set of
covariate points given either a sample from the posterior of the
regression surface or a function \code{FUN(x, params)} and a sample from
the posterior of the parameters.
}
\details{
If design is NULL (default), it is taken to be the identity
matrix of dimension ncol(params), so that the rows of params
are treated as draws from the posterior FUN(x, params).

The 'asymptotic' method assumes that the marginal posteriors of 
the FUN(x, params) are asymptotically normal and is usually
significantly faster and less memory-intensive than the 'quantile'
method, which makes no such assumption.
}
\examples{
### Sample from regression surface posterior
reg.surf.sample <- matrix(rnorm(1000, mean=1:10), ncol=2, byrow=TRUE)
credsubs(reg.surf.sample, cred.level=0.80)

### Parametric case
design <- cbind(1, 1:10)
params <- matrix(rnorm(200, mean=1:2), ncol=2, byrow=TRUE)
credsubs(params, design)

### With custom function
params.sd <- cbind(1 / rgamma(100, 1), params)
FUN.sd <- function(x, params) { params[, -1] \%*\% t(x) / params[, 1] }
credsubs(params.sd, design, FUN.sd, threshold=1)

}
