% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/doubles.R
\name{\%~==\%}
\alias{\%~==\%}
\alias{\%~>=\%}
\alias{\%~>\%}
\alias{\%~<=\%}
\alias{\%~<\%}
\alias{double_equal}
\alias{double_gte}
\alias{double_gt}
\alias{double_lte}
\alias{double_lt}
\title{Relative comparison of double-precision floating point numbers}
\usage{
x \%~==\% y

x \%~>=\% y

x \%~>\% y

x \%~<=\% y

x \%~<\% y

double_equal(x, y, tol = get_tolerance())

double_gte(x, y, tol = get_tolerance())

double_gt(x, y, tol = get_tolerance())

double_lte(x, y, tol = get_tolerance())

double_lt(x, y, tol = get_tolerance())
}
\arguments{
\item{x}{A \link{double} vector.}

\item{y}{A \link{double} vector.}

\item{tol}{A \link{double} vector of tolerances.}
}
\value{
A logical vector.
}
\description{
Fast and efficient methods for comparing floating point numbers
using relative differences.
}
\details{
When either \code{x[i]} or \code{y[i]} contain a number very close to zero,
absolute differences are used, otherwise relative differences are used.

The output of \code{double_equal()} is commutative,
which means the order of arguments don't matter
whereas this is not the case for \code{all.equal.numeric()}.

The calculation is done in C++ and is quite efficient.
Recycling follows the usual R rules and is done without allocating
additional memory.
}
\examples{
library(cppdoubles)

### Basic usage ###

# Standard equality operator
sqrt(2)^2 == 2

# approximate equality operator
sqrt(2)^2 \%~==\% 2

sqrt(2)^2 \%~>=\% 2
sqrt(2)^2 \%~<=\% 2
sqrt(2)^2 \%~>\% 2
sqrt(2)^2 \%~<\% 2

# Alternatively
double_equal(2, sqrt(2)^2)
double_gte(2, sqrt(2)^2)
double_lte(2, sqrt(2)^2)
double_gt(2, sqrt(2)^2)
double_lt(2, sqrt(2)^2)

rel_diff(1, 1 + 2e-10)
double_equal(1, 1 + 2e-10, tol = sqrt(.Machine$double.eps))
double_equal(1, 1 + 2e-10, tol = 1e-10)

# Optionally set a threshold for all comparison
options(cppdoubles.tolerance = 1e-10)
double_equal(1, 1 + 2e-10)

# Floating point errors magnified example

x1 <- 1.1 * 100 * 10^200
x2 <- 110 * 10^200

abs_diff(x1, x2) # Large absolute difference
rel_diff(x1, x2) # Very small relative difference as expected

double_equal(x1, x2)

# all.equal is not commutative but double_equal is
all.equal(10^-8, 2 * 10^-8)
all.equal(2 * 10^-8, 10^-8)

double_equal(10^-8, 2 * 10^-8)
double_equal(2 * 10^-8, 10^-8)

# All comparisons are vectorised and recycled

double_equal(sqrt(1:10),
             sqrt(1:5),
             tol = c(-Inf, 1e-10, Inf))

# One can check for whole numbers like so
whole_number <- function(x, tol = get_tolerance()){
  double_equal(x, round(x))
}
whole_number(seq(-5, 5, 0.25))
}
