% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/readcamtrapDP.R
\name{readcamtrapDP}
\alias{readcamtrapDP}
\title{Convert Camtrap DP format data to camtrapR format}
\usage{
readcamtrapDP(
  deployments_file = "deployments.csv",
  media_file = "media.csv",
  observations_file = "observations.csv",
  datapackage_file = "datapackage.json",
  min_gap_hours = 24,
  removeNA = FALSE,
  removeEmpty = FALSE,
  remove_bbox = TRUE,
  add_file_path = FALSE,
  filter_observations = NULL
)
}
\arguments{
\item{deployments_file}{character. Path to deployments.csv file}

\item{media_file}{character. Path to media.csv file}

\item{observations_file}{character. Path to observations.csv file}

\item{datapackage_file}{character. Path to datapackage.json file}

\item{min_gap_hours}{numeric. Minimum gap in hours to consider as a camera interruption (default: 24)}

\item{removeNA}{logical. Whether to remove columns with only NA values}

\item{removeEmpty}{logical. Whether to remove columns with only empty values}

\item{remove_bbox}{logical. Whether to remove bounding box columns in the observation table}

\item{add_file_path}{logical. Whether to add file path from media table to the observation table}

\item{filter_observations}{Controls which observation types to include. \code{NULL} or \code{FALSE} keeps all observations (default), \code{TRUE} keeps only animal observations, or provide a character vector of specific observation types to include.}
}
\value{
List containing three elements:
\itemize{
  \item CTtable: Data frame with camera trap deployment information in camtrapR format
  \item recordTable: Data frame with species records in camtrapR format
  \item metadata: List containing project metadata extracted from datapackage.json
}
}
\description{
This function converts camera trap data from the Camtrap DP standard format to the format
used by camtrapR. Camtrap DP is an open standard for the FAIR exchange and archiving of 
camera trap data, structured in a simple yet flexible data model consisting of three tables: 
Deployments, Media, and Observations.
}
\details{
Camtrap DP is a standardized format developed under the umbrella of the Biodiversity Information
Standards (TDWG) that facilitates data exchange between different camera trap platforms and systems.
It supports a wide range of camera deployment designs, classification techniques, and analytical use cases.

While the 'camtrapdp' package (available on CRAN) provides general functionality for reading, 
manipulating, and transforming Camtrap DP data, this function specifically converts Camtrap DP 
data directly into the format required by camtrapR, producing the CTtable and recordTable objects 
that camtrapR functions expect.

This function reads the three primary tables from a Camtrap DP dataset:
\itemize{
  \item Deployments: Information about camera trap placements
  \item Observations: Classifications derived from the media files
  \item Media: Information about the media files (images/videos) recorded
}

The Media table is only read if the \code{add_file_path} parameter is set to \code{TRUE}. 

The function converts these into two primary camtrapR data structures:
\itemize{
  \item CTtable: Contains deployment information including station ID, setup/retrieval dates, 
        camera operation problems, camera setup, and more
  \item recordTable: Contains observation records with taxonomic and temporal information
}

Additional features include:
\itemize{
  \item Parsing of deploymentTags and deploymentGroups (in deployments) and observationTags (in observations)
  \item Extraction of taxonomic information from metadata 
  \item Handling of deployment intervals and gaps
}
}
\note{
Camtrap DP was developed as a consensus of a long, in-depth consultation process with 
camera trap data management platforms and major players in the field of camera trapping.
It builds upon the earlier Camera Trap Metadata Standard (CTMS) but addresses its limitations.

The Camtrap DP standard structures data in a way that supports both media-based observations
(using a single media file as source) and event-based observations (considering an event with
a specified duration as source). For the purpose of this function, both are treated to be equivalent. 
Event-based observations are converted to a single timestamp in the recordTable (using the event start time).

Consider using the 'camtrapdp' package (Desmet et al., 2024) for more general operations on 
Camtrap DP data before converting to camtrapR format with this function.
}
\examples{
\dontrun{
# Read a Camtrap DP dataset
camtrap_data <- readcamtrapDP(
  deployments_file = "path/to/deployments.csv",
  media_file = "path/to/media.csv",
  observations_file = "path/to/observations.csv",
  datapackage_file = "path/to/datapackage.json"
)

# alternatively, set the working directory only
setwd("path/to/camtrapdp_data")
camtrap_data <- readcamtrapDP()   # uses default file names


# Extract components
ct_table <- camtrap_data$CTtable
record_table <- camtrap_data$recordTable
metadata <- camtrap_data$metadata

}

}
\references{
Bubnicki, J.W., Norton, B., Baskauf, S.J., et al. (2023). Camtrap DP: an open standard for the 
FAIR exchange and archiving of camera trap data. Remote Sensing in Ecology and Conservation, 10(3), 283-295.

Desmet, P., Govaert, S., Huybrechts, P., Oldoni, D. (2024). camtrapdp: Read and Manipulate Camera 
Trap Data Packages. R package version 0.3.1. https://CRAN.R-project.org/package=camtrapdp
}
\seealso{
\code{\link[camtrapdp]{read_camtrapdp}} in the 'camtrapdp' package for reading Camtrap DP data
\code{\link[camtrapdp]{check_camtrapdp}} in the 'camtrapdp' package for validating Camtrap DP data
}
