#' 
#' Asymptotic least squares estimation
#'
#' @param x the matrix of the right-hand variables (incl. the constant term when 
#' needed).
#' @param y the vector of the left-hand variable.
#' @param omega the covariance matrix of the disturbances.
#' @param ols logical indicating whether to perform OLS (TRUE) or FGLS (FALSE).
#'  The default is FALSE.
#'
#' @return a list with class \code{reg_als} containing \code{"config"}
#'  for the definition of the estimation method and \code{"reg"} for the 
#'  estimation output.
#'  
#' The \code{"config"} data frame includes the following elements:
#' \itemize{
#' \item\bold{family: }"als" (for Asymptotic least squares).
#' \item\bold{method: }"ols" for Ordinary least square, or "fgls" for Feasible
#' generalized least squares.
#' }
#' The list \code{"reg"} includes the following elements (when relevant):
#' \itemize{
#' \item\bold{estim:}a data frame with \code{c_names} for the component names,
#' \code{coef}, the estimated coefficients, \code{std_coef}, the estimated 
#' standard errors, \code{student}the Student statistics for the equality of the 
#' coefficient to 0, \code{p_value}, the p-values of the asymptotic Student test. 
#' \item\bold{cova:} the estimated covariance matrix of the estimator.
#' \item\bold{over_test:}a data frame with the output of the overidentification 
#' test (FGLS only). The statistic is given by \code{stat}, the degrees of
#' freedom by \code{df} and the p-value by \code{p_value}
#' }
#' 
#' @references Chamberlain, G. (1982). Multivariate regression models for panel 
#' data. Journal of econometrics, 18(1), 5-46.
#' Gourieroux, C., Monfort, A., & Trognon, A. (1985). Moindres carrés 
#' asymptotiques. Annales de l'INSEE, 91-122.
#' Kodde, D. A., Plam, F. C., & Pfann, G. A. (1990). Asymptotic least‐squares
#'  estimation efficiency considerations and applications. Journal of Applied 
#'  Econometrics, 5(3), 229-243.
#' 
#' @examples
#' model <- list(c("license"),c("woman"),c("woman","license","inter"))
#' comp <- callback_comp(mobility1,"offer",c("gender","licenses"),"callback",model)
#'     x <- comp$aux_boole
#'     y <- comp$aux_coef
#' omega <- comp$aux_vcov
#'  
#' str(reg_als(x,y,omega))
#' 
#' @importFrom stats pchisq
#' 
#' @export

reg_als <- function(x,y,omega,ols=FALSE){ 
 c_names <- colnames(x)
 if (ols){
 ixx   <- solve(t(x)%*%x)
 e_ols <- ixx%*%t(x)%*%y
 v_ols <- ixx%*%t(x)%*%omega%*%x%*%ixx
 s_ols <- sqrt(diag(v_ols))
 t_ols <- e_ols/s_ols
 p_ols <- 1-pchisq(t_ols^2,1)
 res <- list(estim=data.frame(
                c_names=c_names,
                   coef=e_ols,
               std_coef=s_ols,
                student=t_ols,
                p_value=p_ols),
                   cova=v_ols,
              over_test=NULL)
 } else {#fgls
 iomega <- solve(omega)
   ixox <- solve(t(x)%*%iomega%*%x)
 e_fgls <- ixox%*%t(x)%*%iomega%*%y
 v_fgls <- ixox
 s_fgls <- sqrt(diag(v_fgls))
 t_fgls <- e_fgls/s_fgls
 p_fgls <- 1-pchisq(t_fgls^2,1)
 #overidentification test (with GLS only)
 n_rf <- nrow(x)
 n_sf <- ncol(x)
 r_fgls <- y-x%*%e_fgls
 sover <- ifelse(n_rf==n_sf,0,t(r_fgls)%*%iomega%*%r_fgls)
 dover <- nrow(x)-ncol(x)
 pover <- 1-pchisq(sover,dover)
 over_test <- list(stat=sover,df=dover,p_value=pover)
 res <- list(estim=data.frame(
                c_names=c_names,
                   coef=e_fgls,
               std_coef=s_fgls,
                student=t_fgls,
                p_value=p_fgls),
                   cova=v_fgls,
              over_test=over_test)
 }
#output
    z <- list(config=data.frame(family="als",method=ifelse(ols,"ols","fgls")),
              reg=res)
class(z) <- "reg_als"
return (z)
}

#'
#' Generic regression function
#'
#' @param x A \code{callback_reg} object.
#' @param ... further arguments passed to or from other methods.
#'
#' @export

reg <- function(x, ...) {
  UseMethod("reg")
}

#' 
#' Component model estimation
#'
#' @param x a \code{callback_comp} object.
#' @param method estimation method, "ols" or "fgls" (the default).
#' @param ... further arguments passed to or from other methods.
#'
#' @return a list with class \code{callback_reg} containing \code{"config"}
#'  for the definition of the estimation method and \code{"reg"} for the 
#'  estimation output.
#'  
#' The \code{"config"} data frame includes the following elements:
#' \itemize{
#' \item\bold{family: }"als" (for Asymptotic least squares).
#' \item\bold{method: }"ols" for Ordinary least square, or "fgls" for Feasible
#' generalized least squares.
#' \item\bold{model:}a components model.
#' }
#' The list \code{"reg"} includes the following elements (when relevant):
#' \itemize{
#' \item\bold{estim:}a data frame with \code{c_names} for the component names,
#' \code{coef}, the estimated coefficients, \code{std_coef}, the estimated 
#' standard errors, \code{student}the Student statistics for the equality of the 
#' coefficient to 0, \code{p_value}, the p-values of the asymptotic Student test. 
#' \item\bold{cova:} the estimated covariance matrix of the estimator.
#' \item\bold{over_test:}a data frame with the output of the overidentification 
#' test (FGLS only). The statistic is given by \code{stat}, the degrees of
#' freedom by \code{df} and the p-value by \code{p_value}.
#' }
#' 
#' @references 
#' Duguet E., Le Gall R., L'Horty Y., Petit P. (2018). How does the labour 
#' market history influence the access to hiring interviews? International 
#' Journal of Manpower, 39(4), 519-533, doi: 10.1108/IJM-09-2017-0231.
#' 
#' @examples
#' model <- list(c("license"),c("woman"),c("woman","license","inter"))
#' comp <- callback_comp(data = mobility1, cluster = "offer", 
#' candid = c("gender","licenses"), callback = "callback", model = model)
#' reg(comp)
#' summary(reg(comp))
#' 
#' @export

reg.callback_comp <- function(x,method="fgls",...){
  ols <- ifelse(tolower(method)=="ols",TRUE,FALSE)
  z <- unclass(
    reg_als(x$aux_boole,x$aux_coef,x$aux_vcov,ols=ols)
              )
  z$config$model <- "component model"
  class(z) <- "callback_reg"
  return(z)
}