% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/logistic_reg-fit.R
\name{brulee_logistic_reg}
\alias{brulee_logistic_reg}
\alias{brulee_logistic_reg.default}
\alias{brulee_logistic_reg.data.frame}
\alias{brulee_logistic_reg.matrix}
\alias{brulee_logistic_reg.formula}
\alias{brulee_logistic_reg.recipe}
\title{Fit a logistic regression model}
\usage{
brulee_logistic_reg(x, ...)

\method{brulee_logistic_reg}{default}(x, ...)

\method{brulee_logistic_reg}{data.frame}(
  x,
  y,
  epochs = 20L,
  penalty = 0.001,
  mixture = 0,
  validation = 0.1,
  optimizer = "LBFGS",
  learn_rate = 1,
  momentum = 0,
  batch_size = NULL,
  class_weights = NULL,
  stop_iter = 5,
  verbose = FALSE,
  ...
)

\method{brulee_logistic_reg}{matrix}(
  x,
  y,
  epochs = 20L,
  penalty = 0.001,
  mixture = 0,
  validation = 0.1,
  optimizer = "LBFGS",
  learn_rate = 1,
  momentum = 0,
  batch_size = NULL,
  class_weights = NULL,
  stop_iter = 5,
  verbose = FALSE,
  ...
)

\method{brulee_logistic_reg}{formula}(
  formula,
  data,
  epochs = 20L,
  penalty = 0.001,
  mixture = 0,
  validation = 0.1,
  optimizer = "LBFGS",
  learn_rate = 1,
  momentum = 0,
  batch_size = NULL,
  class_weights = NULL,
  stop_iter = 5,
  verbose = FALSE,
  ...
)

\method{brulee_logistic_reg}{recipe}(
  x,
  data,
  epochs = 20L,
  penalty = 0.001,
  mixture = 0,
  validation = 0.1,
  optimizer = "LBFGS",
  learn_rate = 1,
  momentum = 0,
  batch_size = NULL,
  class_weights = NULL,
  stop_iter = 5,
  verbose = FALSE,
  ...
)
}
\arguments{
\item{x}{Depending on the context:
\itemize{
\item A \strong{data frame} of predictors.
\item A \strong{matrix} of predictors.
\item A \strong{recipe} specifying a set of preprocessing steps
created from \code{\link[recipes:recipe]{recipes::recipe()}}.
}

The predictor data should be standardized (e.g. centered or scaled).}

\item{...}{Options to pass to the learning rate schedulers via
\code{\link[=set_learn_rate]{set_learn_rate()}}. For example, the \code{reduction} or \code{steps} arguments to
\code{\link[=schedule_step]{schedule_step()}} could be passed here.}

\item{y}{When \code{x} is a \strong{data frame} or \strong{matrix}, \code{y} is the outcome
specified as:
\itemize{
\item A \strong{data frame} with 1 factor column (with two levels).
\item A \strong{matrix} with 1 factor column (with two levels).
\item A factor \strong{vector} (with two levels).
}}

\item{epochs}{An integer for the number of epochs of training.}

\item{penalty}{The amount of weight decay (i.e., L2 regularization).}

\item{mixture}{Proportion of Lasso Penalty (type: double, default: 0.0). A
value of mixture = 1 corresponds to a pure lasso model, while mixture = 0
indicates ridge regression (a.k.a weight decay). Must be zero for
optimizers \code{"ADAMw"}, \code{"RMSprop"}, \code{"Adadelta"}.}

\item{validation}{The proportion of the data randomly assigned to a
validation set.}

\item{optimizer}{The method used in the optimization procedure. Possible choices
are \code{"SGD"},  \code{"ADAMw"}, \code{"Adadelta"}, \code{"Adagrad"}, \code{"RMSprop"}, and
\code{"LBFGS"}. \code{"LBFGS"} is the only second-order method, does not use
batches, and is the default.}

\item{learn_rate}{A positive number that controls the initial rapidity that
the model moves along the descent path. Values around 0.1 or less are
typical.}

\item{momentum}{A positive number usually on \verb{[0.50, 0.99]} for the momentum
parameter in gradient descent.  (optimizers \code{"SGD"},  and \code{"RMSprop"} only,
ignored otherwise).}

\item{batch_size}{An integer for the number of training set points in each
batch. (\code{optimizer != "LBFGS"} only, ignored otherwise)}

\item{class_weights}{Numeric class weights (classification only). The value
can be:
\itemize{
\item A named numeric vector (in any order) where the names are the outcome
factor levels.
\item An unnamed numeric vector assumed to be in the same order as the outcome
factor levels.
\item A single numeric value for the least frequent class in the training data
and all other classes receive a weight of one.
}}

\item{stop_iter}{A non-negative integer for how many iterations with no
improvement before stopping.}

\item{verbose}{A logical that prints out the iteration history.}

\item{formula}{A formula specifying the outcome term(s) on the left-hand side,
and the predictor term(s) on the right-hand side.}

\item{data}{When a \strong{recipe} or \strong{formula} is used, \code{data} is specified as:
\itemize{
\item A \strong{data frame} containing both the predictors and the outcome.
}}
}
\value{
A \code{brulee_logistic_reg} object with elements:
\itemize{
\item \code{models_obj}: a serialized raw vector for the torch module.
\item \code{estimates}: a list of matrices with the model parameter estimates per
epoch.
\item \code{best_epoch}: an integer for the epoch with the smallest loss.
\item \code{loss}: A vector of loss values (MSE for regression, negative log-
likelihood for classification) at each epoch.
\item \code{dim}: A list of data dimensions.
\item \code{parameters}: A list of some tuning parameter values.
\item \code{blueprint}: The \code{hardhat} blueprint data.
}
}
\description{
\code{brulee_logistic_reg()} fits a model.
}
\details{
This function fits a linear combination of coefficients and predictors to
model the log odds of the classes. The training process optimizes the
cross-entropy loss function (a.k.a Bernoulli loss).

By default, training halts when the validation loss increases for at least
\code{step_iter} iterations. If \code{validation = 0} the training set loss is used.

The \emph{predictors} data should all be numeric and encoded in the same units (e.g.
standardized to the same range or distribution). If there are factor
predictors, use a recipe or formula to create indicator variables (or some
other method) to make them numeric. Predictors should be in the same units
before training.

The model objects are saved for each epoch so that the number of epochs can
be efficiently tuned. Both the \code{\link[=coef]{coef()}} and \code{\link[=predict]{predict()}} methods for this
model have an \code{epoch} argument (which defaults to the epoch with the best
loss value).

The use of the L1 penalty (a.k.a. the lasso penalty) does \emph{not} force
parameters to be strictly zero (as it does in packages such as \pkg{glmnet}).
The zeroing out of parameters is a specific feature the optimization method
used in those packages.
}
\examples{
\dontshow{if (!brulee:::is_cran_check()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
\donttest{
if (torch::torch_is_installed() & rlang::is_installed(c("recipes", "yardstick", "modeldata"))) {

 library(recipes)
 library(yardstick)

 ## -----------------------------------------------------------------------------
 # increase # epochs to get better results

 data(cells, package = "modeldata")

 cells$case <- NULL

 set.seed(122)
 in_train <- sample(1:nrow(cells), 1000)
 cells_train <- cells[ in_train,]
 cells_test  <- cells[-in_train,]

 # Using matrices
 set.seed(1)
 brulee_logistic_reg(x = as.matrix(cells_train[, c("fiber_width_ch_1", "width_ch_1")]),
                      y = cells_train$class,
                      penalty = 0.10, epochs = 3)

 # Using recipe
 library(recipes)

 cells_rec <-
  recipe(class ~ ., data = cells_train) |>
  # Transform some highly skewed predictors
  step_YeoJohnson(all_numeric_predictors()) |>
  step_normalize(all_numeric_predictors()) |>
  step_pca(all_numeric_predictors(), num_comp = 10)

 set.seed(2)
 fit <- brulee_logistic_reg(cells_rec, data = cells_train,
                             penalty = .01, epochs = 5)
 fit

 autoplot(fit)

 library(yardstick)
 predict(fit, cells_test, type = "prob") |>
  bind_cols(cells_test) |>
  roc_auc(class, .pred_PS)
}
}
\dontshow{\}) # examplesIf}
}
\seealso{
\code{\link[=predict.brulee_logistic_reg]{predict.brulee_logistic_reg()}}, \code{\link[=coef.brulee_logistic_reg]{coef.brulee_logistic_reg()}},
\code{\link[=autoplot.brulee_logistic_reg]{autoplot.brulee_logistic_reg()}}
}
