% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/volumeMetrics.R
\name{deviationSurface}
\alias{deviationSurface}
\alias{referenceSurface}
\alias{blissDeviation}
\alias{blissReference}
\alias{hsaDeviation}
\alias{hsaReference}
\alias{loeweDeviation}
\alias{loeweReference}
\alias{zipDeviation}
\alias{zipReference}
\alias{zipSmoothed}
\title{Non-interacting Reference Surfaces}
\usage{
deviationSurface(concs, act, method = "Bliss", ...)

referenceSurface(concs, act, method = "Bliss", ...)

blissDeviation(concs, act, range, clip = "none")

blissReference(concs, act, range, clip = "none")

hsaDeviation(concs, act, increasing)

hsaReference(concs, act, increasing)

loeweDeviation(concs, act, weights = NULL, limits = c(NA, NA))

loeweReference(concs, act, weights = NULL, limits = c(NA, NA))

zipDeviation(concs, act, range, weights = NULL, limits = c(NA, NA))

zipReference(concs, act, range, weights = NULL, limits = c(NA, NA))

zipSmoothed(concs, act, range, weights = NULL, limits = c(NA, NA))
}
\arguments{
\item{concs}{A width-two array of concentrations representing all measured
dose pairs}

\item{act}{A vector of measured activity or effect values}

\item{method}{A string specifying which model of non-interaction should be
used; possible values are "Bliss" (the default), "HSA", "Loewe", and "ZIP"}

\item{...}{Additional parameters to be passed to the method-specific
deviation or reference surface functions}

\item{range}{For Bliss calculations, the range of effects assumed by Bliss
independence; a  two-element numeric vector containing the minimal effect
and the maximal effect, in that order. For ZIP calculations, the initial
estimate of the minimal and maximal effects used in fitting the individual
dose response curves.}

\item{clip}{Clipping method for Bliss reference and deviation calculations.
Possible values are "pre", "post", and "none".  See details for specifics.}

\item{increasing}{For HSA calculations, is the effect increasing (TRUE)
meaning the "highest" single agent activity is numerically greater; or
decreasing (FALSE), meaning the "highest" single agent activity is
numerically lower. The latter may be appropriate when the modeled response is
target growth or survival.}

\item{weights}{A vector of weights (between 0 and 1) the same length as
the data which determines the weight with which each measurement
will impact the the sum of squared errors.  Weights will be multiplied by
errors \emph{before} squaring.  If \code{NULL} (the default) all weights will be set
to 1. Can be a numeric vector, or the name of a column in \code{data} if \code{formula}
is a symbolic formula}

\item{limits}{For Loewe and ZIP calculations, the fixed values of the
minimal and maximal effects of the drugs and the combination. By default,
both values are set to \code{NA}; any value set to \code{NA} will fit from the data.}
}
\value{
For the deviation functions (\code{deviationSurface}, \code{blissDeviation},
\code{hsaDeviation}, \code{loeweDeviation}, and \code{zipDeviation}), a vector of values
the same length as \code{act} and/or \code{concs} representing the deviation of the
measurement from the specified reference surface.  For the reference
functions (\code{referenceSurface}, \code{blissReference}, \code{hsaReference},
\code{loeweReference}, and \code{zipReferences}), a vector of values the same length
as \code{act} and/or \code{concs} containing the appropriate non-interacting reference
surface itself.  For \code{zipSmoothed}, the smoothed measurement surface given
by ZIP's dose-response-based smoothing method (see Details).
}
\description{
Estimate the best fitting non-interacting reference surface according to
multiple methods, including Loewe, Bliss, and HSA
}
\details{
This collection of functions can be used to implement a family of combination
analysis methods known as "deviation" methods.  The details of the methods
differ, but the core strategy is common to all of them: estimate what a given
response measurement \emph{would be} based on individual behaviors and some model
of non-interaction, and use the deviation of the measured response from that
expected response as a measure of the degree of synergy or antagonism.

Bliss independence is the most widely used of these, and can be described as
the assumption that any response represents a fraction of the target
population being unaffected, and that a combined response corresponds to the
product of these fractions for both drugs (Bliss, 1939). It is extremely simple to
calculate, and relies on the intuitive model of probabilisticalyly independent
events. Because it treats responses as a scaled representation of
probabilities, it requires that all values be expressed relative to two
limiting values: the response seen when \strong{all} targets are unaffected (the
minimal effect) and the response seen when \strong{none} of the targets remain
unaffected (the maximal effect).  For a Bliss independent surface to be
estimated, these two values must be provided, using the parameter \code{range}.
Further, because values outside of this range would represent proportions
above 1 or below 0, most Bliss calculations involve some adjustment of the
data to ensure they always lie within the specified range.  The Bliss
functions support two ways of doing this to generate a reference surface:
"pre" will clip all values to the range immediately; "post" will clip all
calculated responses to the given range after they have been combined. A
third option, "none", performs no clipping at all, and allows for proportions
outside of the 0 to 1 range.

The highest-single-agent, or HSA model, is even simpler than Bliss. The
effect of a combined pair of doses is simply the "greater" of the individual
effects produced by the two drugs at those levels.  The word "greater" is
placed in quotes here as taking the larger response value is only appropriate
when a numerically larger measurement corresponds to greater activity. For
survival or growth inhibition studies, the reverse may be true; the parameter
\code{increasing} allows the user to specify this directionality.

Loewe additivity is one of the oldest models of non-interaction, and the
inspiration for BRAID additivity (Loewe and Muischnek, 1926).  According to
Loewe additivity, the combined response to a pair of doses the the effect
such that the two  individual doses represent complementary fractions of the
individual doses of both drugs that produce the same effect in isolation. It
is considered the gold standard of non-interaction by many researchers, but =
has several significant pragmatic disadvantages.  It requires that the full
dose response behavior of both drugs is known, and that they produce an
identical range of effects.  The \code{loeweReference} and \code{loeweDeviation}
functions perform basic dose response fitting with the additional constraint
of matching minimal and maximal effects, either fixed by the \code{limits}
parameter or estimated directly from the data.

The zero-interaction potency, or ZIP, model is a variant of Bliss
independence that uses smoothing to give more robust values (Wooten \emph{et al.}
2015). The reference surface is calculated by fitting the dose response of
the individual drugs and then combining them using Bliss independence; the
method then adds additional robustness and smooths the measured surface
itself by fitting each constant-dose set of data points as its own dose
response curve, constrained to match the other drug's effect when the first
drug is zero.  Fitting these partial dose-response curves in either
direction produces a smoothed version of the original measurements (which
can be accessed directly using the function \code{zipSmoothed}), from which the
reference surface is subtracted to get the deviation (or "delta") surface.
}
\examples{
surface <- additiveExample
concs1 <- cbind(surface$concA,surface$concB)
act1 <- surface$measure

sum(deviationSurface(concs1,act1,"Bliss",range=c(0,1)))
sum(deviationSurface(concs1,act1,"Loewe"))
surface$hsa <- hsaReference(concs1,act1,increasing=TRUE)

surface <- synergisticExample
concs2 <- cbind(surface$concA,surface$concB)
act2 <- surface$measure

sum(deviationSurface(concs2,act2,"ZIP",range=c(0,1)))
sum(deviationSurface(concs2,act2,"Loewe"))
surface$smooth <- zipSmoothed(concs2,act2,range=c(0,1))
}
\references{
Bliss, Chester I. 1939. “The Toxicity of Poisons Applied Jointly 1.”
\emph{Annals of Applied Biology} \strong{26} (3): 585–615.

Loewe, S, and H Muischnek. 1926. “Uber Kombinationswirkungen.”
Naunyn. Schmiedebergs. Arch. Pharmacol. 114: 313–26.

Yadav, Bhagwan, Krister Wennerberg, Tero Aittokallio, and Jing Tang. 2015.
“Searching for Drug Synergy in Complex Dose–Response Landscapes Using an
Interaction Potency Model.”
\emph{Computational and Structural Biotechnology Journal} \strong{13}: 504–13.
}
