% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/transfer-entropy.R
\name{run_transfer_entropy}
\alias{run_transfer_entropy}
\title{Transfer Entropy for Counts, Rates, and Binary Series}
\usage{
run_transfer_entropy(
  DT,
  lags = 1:3,
  shuffles = 1000,
  seed = 123,
  use_progress = TRUE
)
}
\arguments{
\item{DT}{A \code{data.table} or \code{data.frame} containing at least
the following columns:
\itemize{
\item \code{I}, \code{C}: count variables (non-negative integers).
\item \code{exposure50}: exposure used to form rates (must be
strictly positive).
\item \code{log_exposure50}: log of the exposure (offset).
\item \code{t_norm}, \code{Regime}, \code{EconCycle},
\code{PopDensity}, \code{Epidemics}, \code{Climate}, \code{War}:
covariates used by the pre-whitening GLMs.
}}

\item{lags}{Integer vector of lag orders \code{L} for which transfer
entropy is computed (passed to \code{lx} and \code{ly} in
\code{RTransferEntropy::transfer_entropy()}).}

\item{shuffles}{Integer; number of shuffle replications for the
surrogate-distribution-based significance test in
\code{transfer_entropy()}.}

\item{seed}{Integer; base random seed used for reproducibility of the
pre-whitening and transfer entropy computations.}

\item{use_progress}{Logical; reserved for future use to toggle progress
reporting. Currently not used.}
}
\value{
A \code{data.frame} with one row per lag and type, and columns:
\itemize{
\item \code{lag}: lag order used in \code{transfer_entropy()}.
\item \code{TE_ItoC}, \code{p_ItoC}: transfer entropy and p-value
from \code{I} to \code{C}.
\item \code{TE_CtoI}, \code{p_CtoI}: transfer entropy and p-value
from \code{C} to \code{I}.
\item \code{type}: transformation used (\code{"counts"},
\code{"rates"}, or \code{"binary"}).
}
}
\description{
Computes pairwise transfer entropy between \code{I} and \code{C} for
three transformations of the data: raw counts, rates (count/exposure),
and binary presence/absence. Each series is first pre-whitened via a
GLM and transfer entropy is then estimated for a grid of lags using
\pkg{RTransferEntropy}. Results are written to separate CSV files and
to a combined summary.
}
\details{
The function proceeds in four steps:
\enumerate{
\item \strong{Counts}: \code{I} and \code{C} are pre-whitened via
\code{\link{prewhiten_count_glm}} (Negative Binomial with offset and
Poisson fallback). Transfer entropy is computed in both directions
(\code{I→C} and \code{C→I}) for each lag in \code{lags}. Results are
saved to \code{"transfer_entropy_counts.csv"}.
\item \strong{Rates}: \code{I} and \code{C} are divided by
\code{exposure50}, pre-whitened via
\code{\link{prewhiten_rate_glm}}, and transfer entropy is recomputed.
Results are saved to \code{"transfer_entropy_rates.csv"}. A check is
performed to ensure \code{exposure50 > 0} for all observations.
\item \strong{Binary}: \code{I} and \code{C} are recoded as 0/1
presence/absence indicators and pre-whitened via
\code{\link{prewhiten_bin_glm}}. Transfer entropy is computed again
and results are saved to \code{"transfer_entropy_binary.csv"}.
\item \strong{Combined}: All tables are stacked into a single data
frame with a \code{type} column (\code{"counts"}, \code{"rates"},
\code{"binary"}) and written to \code{"transfer_entropy.csv"}.
}

Internally, the helpers \code{\link{.get_stat}} and
\code{\link{.get_pval}} are used to extract the transfer entropy
statistic and p-value from the objects returned by
\code{RTransferEntropy::transfer_entropy()}. The function assumes a
global \code{dir_csv} object (character scalar) indicating the output
directory for CSV files.
}
\examples{
\donttest{
library(data.table)

# 1. Create dummy data with ALL covariates required by prewhiten_*_glm()
# The internal GLM formulas likely include:
# I ~ t_norm + Regime + EconCycle + PopDensity + Epidemics + Climate + War
DT <- data.table(
  year = 2000:2029,
  I = rpois(30, lambda = 10),
  C = rpois(30, lambda = 8),
  exposure50 = runif(30, 100, 200),
  log_exposure50 = log(runif(30, 100, 200)),
  # Covariates
  t_norm = seq(-1, 1, length.out = 30),
  Regime = factor(sample(c("A", "B"), 30, replace = TRUE)),
  EconCycle = rnorm(30),
  PopDensity = rnorm(30),
  Epidemics = rnorm(30),
  Climate = rnorm(30),
  War = rnorm(30)
)

# 2. Define global paths using tempdir() (Fixes CRAN policy)
# run_transfer_entropy writes output to 'dir_csv'
tmp_dir <- tempdir()
dir_csv <- file.path(tmp_dir, "csv")
if (!dir.exists(dir_csv)) dir.create(dir_csv, recursive = TRUE)

# 3. Run the function
# Using fewer shuffles for a faster example check
te_tab <- run_transfer_entropy(DT, lags = 1, shuffles = 10, seed = 123)

# Inspect results
if (!is.null(te_tab)) {
  print(subset(te_tab, type == "counts"))
}
}

}
