% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fdt.R
\name{fdt}
\alias{fdt}
\title{Format a datetime with a formatting string}
\usage{
fdt(input, format = NULL, use_tz = NULL, locale = NULL)
}
\arguments{
\item{input}{A vector of date, time, or datetime values. Several
representations are acceptable here including strings, \code{Date} objects, or
\code{POSIXct} objects. Refer to the \emph{Valid Input Values} section for more
information.}

\item{format}{The formatting string to apply to all \code{input} values. If not
provided, the inputs will be formatted to ISO 8601 datetime strings. The
\emph{Date/Time Format Syntax} section has detailed information on how to create
a formatting string.}

\item{use_tz}{A tzid (e.g., \code{"America/New_York"}) time-zone designation for
precise formatting of related outputs. This overrides any time zone
information available in \code{character}-based input values and is applied to
all vector components.}

\item{locale}{The output locale to use for formatting the input value
according to the specified locale's rules. Example locale names include
\code{"en"} for English (United States) and \code{"es-EC"} for Spanish (Ecuador). If
a locale isn't provided the \code{"en"} locale will be used. The
\link{fdt_locales_vec} vector contains the valid locales and \link{fdt_locales_lst}
list provides an easy way to obtain a valid locale.}
}
\value{
A character vector of formatted dates, times, or datetimes.
}
\description{
With \code{fdt()}, we can format datetime values with the greatest of ease, and,
with great power. There is a lot of leniency in what types of input
date/time/datetime values can be passed in. The formatting string allows for
a huge array of possibilities when formatting. Not only that, we can set a
\code{locale} value and get the formatted values localized in the language/region
of choice. There's plenty of ways to represent time zone information, and
this goes along with the option to enrich the input values with a precise
time zone identifier (like \code{"America/Los_Angeles"}). The choices are ample
here, with the goal being a comprehensiveness and ease-of-use in date/time
formatting.
}
\section{Valid Input Values}{


The \code{input} argument of the \code{fdt()} function allows for some flexibility on
what can be passed in. This section describes the kinds of inputs that are
understandable by \code{fdt()}. A vector of strings is allowed, as are vectors of
\code{Date} or \code{POSIXct} values.

If using strings, a good option is to use those that adhere to the ISO
8601:2004 standard. For a datetime this can be of the form
\verb{YYYY-MM-DDThh:mm:ss.s<TZD>}. With this, \code{YYYY-MM-DD} corresponds to the
date, the literal \code{"T"} is optional, \code{hh:mm:ss} is the time (where seconds,
\code{ss}, is optional as are \code{.s} for fractional seconds), and \verb{<TZD>} refers to
an optional time-zone designation (more on time zones in the next paragraph).
You can provide just the date part, and this assumes midnight as an implicit
time. It's also possible to provide just the time part, and this internally
assembles a datetime that uses the current date. When formatting standalone
dates or times, you'll probably just format the explicit parts but \code{fdt()}
won't error if you format the complementary parts.

The time zone designation on string-based datetimes is completely optional.
If not provided then \code{"UTC"} is assumed. If you do want to supply time zone
information, it can be given as an offset value with the following
constructions:
\itemize{
\item \verb{<time>Z}
\item \verb{<time>(+/-)hh:mm}
\item \verb{<time>(+/-)hhmm}
\item \verb{<time>(+/-)hh}
}

The first, \verb{<time>Z}, is zone designator for the zero UTC offset; it's
equivalent to \code{"+00:00"}. The next two are formats for providing the time
offsets from UTC with hours and minutes fields. Examples are \code{"-05:00"} (New
York, standard time), \code{"+0200"} (Cairo), and \code{"+05:30"} (Mumbai). Note that
the colon is optional but leading zeros to maintain two-digit widths are
essential. The final format, \verb{<time>(+/-)hh}, omits the minutes field and as
so many offsets have \code{"00"} minutes values, this can be convenient.

We can also supply an Olson/IANA-style time zone identifier (tzid) in
parentheses within the string, or, as a value supplied to \code{use_tz} (should a
tzid apply to all date/time/datetime values in the \code{input} vector). By
extension, this would use the form: \verb{YYYY-MM-DDThh:mm:ss.s<TZD>(<tzid>)}.
Both a \verb{<TZD>} (UTC offset value) and a \verb{<tzid>} shouldn't really be used
together but if that occurs the \verb{<tzid>} overrides the UTC offset. Here are
some examples:
\itemize{
\item \code{"2018-07-04T22:05 (America/Vancouver)"} (preferable)
\item \code{"2018-07-04T22:05-0800(America/Vancouver)"} (redundant, but still okay)
}

A tzid contains much more information about the time zone than a UTC offset
value since it is tied to some geographical location and the timing of
Standard Time (STD) and Daylight Saving Time (DST) is known. In essence we
can derive UTC offset values from a tzid and also a host of other identifiers
(time zone names, their abbreviations, etc.). Here are some examples of valid
tzid values that can be used:
\itemize{
\item \code{"America/Jamaica"} (the official time in Jamaica, or, \code{"Jamaica Time"})
\item \code{"Australia/Perth"} (\code{"+08:00"} year round in Western Australia)
\item \code{"Europe/Dublin"} (IST/GMT time: \code{"+01:00"}/\code{"+00:00"})
}

The tz database (a compilation of information about the world's time zones)
consists of canonical zone names (those that are primary and preferred) and
alternative names (less preferred in modern usage, and was either discarded
or more commonly replaced by a canonical zone name). The \code{fdt()} function can
handle both types and what occurs is that non-canonical tzid values are
internally mapped onto canonical zone names. Here's a few examples:
\itemize{
\item \code{"Africa/Luanda"} (in Angola) maps to \code{"Africa/Lagos"}
\item \code{"America/Indianapolis"} maps to \code{"America/Indiana/Indianapolis"}
\item \code{"Asia/Calcutta"} maps to \code{"Asia/Kolkata"}
\item \code{"Pacific/Midway"} maps to \code{"Pacific/Pago_Pago"}
\item \code{"Egypt"} maps to \code{"Africa/Cairo"}
}

For the most part, the Olson-format tzid follows the form \code{"{region}/{city}"}
where the region is usually a continent, the city is considered an 'exemplar
city', and the exemplar city itself belongs in a country.
}

\section{Date/Time Format Syntax}{


A formatting pattern as used in \strong{bigD} consists of a string of characters,
where certain strings are replaced with date and time data that are derived
from the parsed input.

The characters used in patterns are tabulated below to show which specific
strings produce which outputs (e.g., \code{"y"} for the year). A common pattern is
characters that are used consecutively to produce variations on a date, time,
or timezone output. Say that the year in the input is 2015. If using \code{"yy"}
you'll get \code{"15"} but with \code{"yyyy"} the output becomes \code{"2015"}. There's a
whole lot of this, so the following subsections try to illustrate as best as
possible what each string will produce. All of the examples will use this
string-based datetime input unless otherwise indicated:

\code{"2018-07-04T22:05:09.2358(America/Vancouver)"}
\subsection{Year}{
\subsection{Calendar Year (little y)}{

This yields the calendar year, which is always numeric. In most cases the
length of the \code{"y"} field specifies the minimum number of digits to display,
zero-padded as necessary. More digits will be displayed if needed to show the
full year. There is an exception: \code{"yy"} gives use just the two low-order
digits of the year, zero-padded as necessary. For most use cases, \code{"y"} or
\code{"yy"} should be good enough.\tabular{ll}{
   Field Patterns \tab Output \cr
   \code{"y"} \tab \code{"2018"} \cr
   \code{"yy"} \tab \code{"18"} \cr
   \code{"yyy"} to \code{"yyyyyyyyy"} \tab \code{"2018"} to \code{"000002018"} \cr
}

}

\subsection{Year in the Week in Year Calendar (big Y)}{

This is the year in 'Week of Year' based calendars in which the year
transition occurs on a week boundary. This may differ from calendar year
\code{"y"} near a year transition. This numeric year designation is used in
conjunction with pattern character \code{"w"} in the ISO year-week calendar as
defined by ISO 8601.\tabular{ll}{
   Field Patterns \tab Output \cr
   \code{"Y"} \tab \code{"2018"} \cr
   \code{"YY"} \tab \code{"18"} \cr
   \code{"YYY"} to \code{"YYYYYYYYY"} \tab \code{"2018"} to \code{"000002018"} \cr
}

}

}

\subsection{Quarter}{
\subsection{Quarter of the Year: formatting (big Q) and standalone (little q)}{

The quarter names are identified numerically, starting at \code{1} and ending at
\code{4}. Quarter names may vary along two axes: the width and the context. The
context is either 'formatting' (taken as a default), which the form used
within a complete date format string, or, 'standalone', the form for date
elements used independently (such as in calendar headers). The standalone
form may be used in any other date format that shares the same form of the
name. Here, the formatting form for quarters of the year consists of some run
of \code{"Q"} values whereas the standalone form uses \code{"q"}.\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"Q"}/\code{"q"} \tab \code{"3"} \tab Numeric, one digit \cr
   \code{"QQ"}/\code{"qq"} \tab \code{"03"} \tab Numeric, two digits (zero padded) \cr
   \code{"QQQ"}/\code{"qqq"} \tab \code{"Q3"} \tab Abbreviated \cr
   \code{"QQQQ"}/\code{"qqqq"} \tab \code{"3rd quarter"} \tab Wide \cr
   \code{"QQQQQ"}/\code{"qqqqq"} \tab \code{"3"} \tab Narrow \cr
}

}

}

\subsection{Month}{
\subsection{Month: formatting (big M) and standalone (big L)}{

The month names are identified numerically, starting at \code{1} and ending at
\code{12}. Month names may vary along two axes: the width and the context. The
context is either 'formatting' (taken as a default), which the form used
within a complete date format string, or, 'standalone', the form for date
elements used independently (such as in calendar headers). The standalone
form may be used in any other date format that shares the same form of the
name. Here, the formatting form for months consists of some run of \code{"M"}
values whereas the standalone form uses \code{"L"}.\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"M"}/\code{"L"} \tab \code{"7"} \tab Numeric, minimum digits \cr
   \code{"MM"}/\code{"LL"} \tab \code{"07"} \tab Numeric, two digits (zero padded) \cr
   \code{"MMM"}/\code{"LLL"} \tab \code{"Jul"} \tab Abbreviated \cr
   \code{"MMMM"}/\code{"LLLL"} \tab \code{"July"} \tab Wide \cr
   \code{"MMMMM"}/\code{"LLLLL"} \tab \code{"J"} \tab Narrow \cr
}

}

}

\subsection{Week}{
\subsection{Week of Year (little w)}{

Values calculated for the week of year range from \code{1} to \code{53}. Week \code{1} for a
year is the first week that contains at least the specified minimum number of
days from that year. Weeks between week \code{1} of one year and week \code{1} of the
following year are numbered sequentially from \code{2} to \code{52} or \code{53} (if
needed).

There are two available field lengths. Both will display the week of year
value but the \code{"ww"} width will always show two digits (where weeks \code{1} to
\code{9} are zero padded).\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"w"} \tab \code{"27"} \tab Minimum digits \cr
   \code{"ww"} \tab \code{"27"} \tab Two digits (zero padded) \cr
}

}

\subsection{Week of Month (big W)}{

The week of a month can range from \code{1} to \code{5}. The first day of every month
always begins at week \code{1} and with every transition into the beginning of a
week, the week of month value is incremented by \code{1}.\tabular{ll}{
   Field Pattern \tab Output \cr
   \code{"W"} \tab \code{"1"} \cr
}

}

}

\subsection{Day}{
\subsection{Day of Month (little d)}{

The day of month value is always numeric and there are two available field
length choices in its formatting. Both will display the day of month value
but the \code{"dd"} formatting will always show two digits (where days \code{1} to \code{9}
are zero padded).\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"d"} \tab \code{"4"} \tab Minimum digits \cr
   \code{"dd"} \tab \code{"04"} \tab Two digits, zero padded \cr
}

}

\subsection{Day of Year (big D)}{

The day of year value ranges from \code{1} (January 1) to either \code{365} or \code{366}
(December 31), where the higher value of the range indicates that the year is
a leap year (29 days in February, instead of 28). The field length specifies
the minimum number of digits, with zero-padding as necessary.\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"D"} \tab \code{"185"} \tab  \cr
   \code{"DD"} \tab \code{"185"} \tab Zero padded to minimum width of 2 \cr
   \code{"DDD"} \tab \code{"185"} \tab Zero padded to minimum width of 3 \cr
}

}

\subsection{Day of Week in Month (big F)}{

The day of week in month returns a numerical value indicating the number of
times a given weekday had occurred in the month (e.g., '2nd Monday in
March'). This conveniently resolves to predicable case structure where ranges
of day of the month values return predictable day of week in month values:
\itemize{
\item days \code{1} - \code{7} -> \code{1}
\item days \code{8} - \code{14} -> \code{2}
\item days \code{15} - \code{21} -> \code{3}
\item days \code{22} - \code{28} -> \code{4}
\item days \code{29} - \code{31} -> \code{5}
}\tabular{ll}{
   Field Pattern \tab Output \cr
   \code{"F"} \tab \code{"1"} \cr
}

}

\subsection{Modified Julian Date (little g)}{

The modified version of the Julian date is obtained by subtracting
2,400,000.5 days from the Julian date (the number of days since January 1,
4713 BC). This essentially results in the number of days since midnight
November 17, 1858. There is a half day offset (unlike the Julian date, the
modified Julian date is referenced to midnight instead of noon).\tabular{ll}{
   Field Patterns \tab Output \cr
   \code{"g"} to \code{"ggggggggg"} \tab \code{"58303"} -> \code{"000058303"} \cr
}

}

}

\subsection{Weekday}{
\subsection{Day of Week Name (big E)}{

The name of the day of week is offered in four different widths.\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"E"}, \code{"EE"}, or \code{"EEE"} \tab \code{"Wed"} \tab Abbreviated \cr
   \code{"EEEE"} \tab \code{"Wednesday"} \tab Wide \cr
   \code{"EEEEE"} \tab \code{"W"} \tab Narrow \cr
   \code{"EEEEEE"} \tab \code{"We"} \tab Short \cr
}

}

}

\subsection{Periods}{
\subsection{AM/PM Period of Day (little a)}{

This denotes before noon and after noon time periods. May be upper or
lowercase depending on the locale and other options. The wide form may be
the same as the short form if the 'real' long form (e.g. 'ante meridiem') is
not customarily used. The narrow form must be unique, unlike some other
fields.\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"a"}, \code{"aa"}, or \code{"aaa"} \tab \code{"PM"} \tab Abbreviated \cr
   \code{"aaaa"} \tab \code{"PM"} \tab Wide \cr
   \code{"aaaaa"} \tab \code{"p"} \tab Narrow \cr
}

}

\subsection{AM/PM Period of Day Plus Noon and Midnight (little b)}{

Provide AM and PM as well as phrases for exactly noon and midnight. May be
upper or lowercase depending on the locale and other options. If the locale
doesn't have the notion of a unique 'noon' (i.e., 12:00), then the PM form
may be substituted. A similar behavior can occur for 'midnight' (00:00) and
the AM form. The narrow form must be unique, unlike some other fields.

(a) \code{input_midnight}: \code{"2020-05-05T00:00:00"}
(b) \code{input_noon}: \code{"2020-05-05T12:00:00"}\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"b"}, \code{"bb"}, or \code{"bbb"} \tab (a) \code{"midnight"} \tab Abbreviated \cr
    \tab (b) \code{"noon"} \tab  \cr
   \code{"bbbb"} \tab (a) \code{"midnight"} \tab Wide \cr
    \tab (b) \code{"noon"} \tab  \cr
   \code{"bbbbb"} \tab (a) \code{"mi"} \tab Narrow \cr
    \tab (b) \code{"n"} \tab  \cr
}

}

\subsection{Flexible Day Periods (big B)}{

Flexible day periods denotes things like 'in the afternoon', 'in the
evening', etc., and the flexibility comes from a locale's language and
script. Each locale has an associated rule set that specifies when the day
periods start and end for that locale.

(a) \code{input_morning}: \code{"2020-05-05T00:08:30"}
(b) \code{input_afternoon}: \code{"2020-05-05T14:00:00"}\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"B"}, \code{"BB"}, or \code{"BBB"} \tab (a) \code{"in the morning"} \tab Abbreviated \cr
    \tab (b) \code{"in the afternoon"} \tab  \cr
   \code{"BBBB"} \tab (a) \code{"in the morning"} \tab Wide \cr
    \tab (b) \code{"in the afternoon"} \tab  \cr
   \code{"BBBBB"} \tab (a) \code{"in the morning"} \tab Narrow \cr
    \tab (b) \code{"in the afternoon"} \tab  \cr
}

}

}

\subsection{Hours, Minutes, and Seconds}{
\subsection{Hour 0-23 (big H)}{

Hours from \code{0} to \code{23} are for a standard 24-hour clock cycle (midnight plus
1 minute is \code{00:01}) when using \code{"HH"} (which is the more common width that
indicates zero-padding to 2 digits).

Using \code{"2015-08-01T08:35:09"}:\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"H"} \tab \code{"8"} \tab Numeric, minimum digits \cr
   \code{"HH"} \tab \code{"08"} \tab Numeric, 2 digits (zero padded) \cr
}

}

\subsection{Hour 1-12 (little h)}{

Hours from \code{1} to \code{12} are for a standard 12-hour clock cycle (midnight plus
1 minute is \code{12:01}) when using \code{"hh"} (which is the more common width that
indicates zero-padding to 2 digits).

Using \code{"2015-08-01T08:35:09"}:\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"h"} \tab \code{"8"} \tab Numeric, minimum digits \cr
   \code{"hh"} \tab \code{"08"} \tab Numeric, 2 digits (zero padded) \cr
}

}

\subsection{Hour 1-24 (little k)}{

Using hours from \code{1} to \code{24} is a less common way to express a 24-hour clock
cycle (midnight plus 1 minute is \code{24:01}) when using \code{"kk"} (which is the
more common width that indicates zero-padding to 2 digits).

Using \code{"2015-08-01T08:35:09"}:\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"k"} \tab \code{"9"} \tab Numeric, minimum digits \cr
   \code{"kk"} \tab \code{"09"} \tab Numeric, 2 digits (zero padded) \cr
}

}

\subsection{Hour 0-11 (big K)}{

Using hours from \code{0} to \code{11} is a less common way to express a 12-hour clock
cycle (midnight plus 1 minute is \code{00:01}) when using \code{"KK"} (which is the
more common width that indicates zero-padding to 2 digits).

Using \code{"2015-08-01T08:35:09"}:\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"K"} \tab \code{"7"} \tab Numeric, minimum digits \cr
   \code{"KK"} \tab \code{"07"} \tab Numeric, 2 digits (zero padded) \cr
}

}

\subsection{Minute (little m)}{

The minute of the hour which can be any number from \code{0} to \code{59}. Use \code{"m"} to
show the minimum number of digits, or \code{"mm"} to always show two digits
(zero-padding, if necessary).\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"m"} \tab \code{"5"} \tab Numeric, minimum digits \cr
   \code{"mm"} \tab \code{"06"} \tab Numeric, 2 digits (zero padded) \cr
}

}

\subsection{Seconds (little s)}{

The second of the minute which can be any number from \code{0} to \code{59}. Use \code{"s"}
to show the minimum number of digits, or \code{"ss"} to always show two digits
(zero-padding, if necessary).\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"s"} \tab \code{"9"} \tab Numeric, minimum digits \cr
   \code{"ss"} \tab \code{"09"} \tab Numeric, 2 digits (zero padded) \cr
}

}

\subsection{Fractional Second (big S)}{

The fractional second truncates (like other time fields) to the width
requested (i.e., count of letters). So using pattern \code{"SSSS"} will display
four digits past the decimal (which, incidentally, needs to be added manually
to the pattern).\tabular{ll}{
   Field Patterns \tab Output \cr
   \code{"S"} to \code{"SSSSSSSSS"} \tab \code{"2"} -> \code{"235000000"} \cr
}

}

\subsection{Milliseconds Elapsed in Day (big A)}{

There are 86,400,000 milliseconds in a day and the \code{"A"} pattern will provide
the whole number. The width can go up to nine digits with \code{"AAAAAAAAA"} and
these higher field widths will result in zero padding if necessary.

Using \code{"2011-07-27T00:07:19.7223"}:\tabular{ll}{
   Field Patterns \tab Output \cr
   \code{"A"} to \code{"AAAAAAAAA"} \tab \code{"439722"} -> \code{"000439722"} \cr
}

}

}

\subsection{Era}{
\subsection{The Era Designator (big G)}{

This provides the era name for the given date. The Gregorian calendar has two
eras: AD and BC. In the AD year numbering system, AD 1 is immediately
preceded by 1 BC, with nothing in between them (there was no year zero).\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"G"}, \code{"GG"}, or \code{"GGG"} \tab \code{"AD"} \tab Abbreviated \cr
   \code{"GGGG"} \tab \code{"Anno Domini"} \tab Wide \cr
   \code{"GGGGG"} \tab \code{"A"} \tab Narrow \cr
}

}

}

\subsection{Time Zones}{
\subsection{TZ // Short and Long Specific non-Location Format (little z)}{

The short and long specific non-location formats for time zones are suggested
for displaying a time with a user friendly time zone name. Where the short
specific format is unavailable, it will fall back to the short localized GMT
format (\code{"O"}). Where the long specific format is unavailable, it will fall
back to the long localized GMT format (\code{"OOOO"}).\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"z"}, \code{"zz"}, or \code{"zzz"} \tab \code{"PDT"} \tab Short Specific \cr
   \code{"zzzz"} \tab \code{"Pacific Daylight Time"} \tab Long Specific \cr
}

}

\subsection{TZ // Common UTC Offset Formats (big Z)}{

The ISO8601 basic format with hours, minutes and optional seconds fields is
represented by \code{"Z"}, \code{"ZZ"}, or \code{"ZZZ"}. The format is equivalent to RFC 822
zone format (when the optional seconds field is absent). This is equivalent
to the \code{"xxxx"} specifier. The field pattern \code{"ZZZZ"} represents the long
localized GMT format. This is equivalent to the \code{"OOOO"} specifier. Finally,
\code{"ZZZZZ"} pattern yields the ISO8601 extended format with hours, minutes and
optional seconds fields. The ISO8601 UTC indicator \code{Z} is used when local
time offset is \code{0}. This is equivalent to the \code{"XXXXX"} specifier.\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"Z"}, \code{"ZZ"}, or \code{"ZZZ"} \tab \code{"-0700"} \tab ISO 8601 basic format \cr
   \code{"ZZZZ"} \tab \code{"GMT-7:00"} \tab Long localized GMT format \cr
   \code{"ZZZZZ"} \tab \code{"-07:00"} \tab ISO 8601 extended format \cr
}

}

\subsection{TZ // Short and Long Localized GMT Formats (big O)}{

The localized GMT formats come in two widths \code{"O"} (which removes the minutes
field if it's \code{0}) and \code{"OOOO"} (which always contains the minutes field).
The use of the \code{GMT} indicator changes according to the locale.\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"O"} \tab \code{"GMT-7"} \tab Short localized GMT format \cr
   \code{"OOOO"} \tab \code{"GMT-07:00"} \tab Long localized GMT format \cr
}

}

\subsection{TZ // Short and Long Generic non-Location Formats (little v)}{

The generic non-location formats are useful for displaying a recurring wall
time (e.g., events, meetings) or anywhere people do not want to be overly
specific. Where either of these is unavailable, there is a fallback to the
generic location format (\code{"VVVV"}), then the short localized GMT format as
the final fallback.\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"v"} \tab \code{"PT"} \tab Short generic non-location format \cr
   \code{"vvvv"} \tab \code{"Pacific Time"} \tab Long generic non-location format \cr
}

}

\subsection{TZ // Short Time Zone IDs and Exemplar City Formats (big V)}{

These formats provide variations of the time zone ID and often include the
exemplar city. The widest of these formats, \code{"VVVV"}, is useful for
populating a choice list for time zones, because it supports 1-to-1 name/zone
ID mapping and is more uniform than other text formats.\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"V"} \tab \code{"cavan"} \tab Short time zone ID \cr
   \code{"VV"} \tab \code{"America/Vancouver"} \tab Long time zone ID \cr
   \code{"VVV"} \tab \code{"Vancouver"} \tab The tz exemplar city \cr
   \code{"VVVV"} \tab \code{"Vancouver Time"} \tab Generic location format \cr
}

}

\subsection{TZ // ISO 8601 Formats with Z for +0000 (big X)}{

The \code{"X"}-\code{"XXX"} field patterns represent valid ISO 8601 patterns for time
zone offsets in datetimes. The final two widths, \code{"XXXX"} and \code{"XXXXX"} allow
for optional seconds fields. The seconds field is \emph{not} supported by the ISO
8601 specification. For all of these, the ISO 8601 UTC indicator \code{Z} is used
when the local time offset is \code{0}.\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"X"} \tab \code{"-07"} \tab ISO 8601 basic format (h, optional m) \cr
   \code{"XX"} \tab \code{"-0700"} \tab ISO 8601 basic format (h & m) \cr
   \code{"XXX"} \tab \code{"-07:00"} \tab ISO 8601 extended format (h & m) \cr
   \code{"XXXX"} \tab \code{"-0700"} \tab ISO 8601 basic format (h & m, optional s) \cr
   \code{"XXXXX"} \tab \code{"-07:00"} \tab ISO 8601 extended format (h & m, optional s) \cr
}

}

\subsection{TZ // ISO 8601 Formats (no use of Z for +0000) (little x)}{

The \code{"x"}-\code{"xxxxx"} field patterns represent valid ISO 8601 patterns for time
zone offsets in datetimes. They are similar to the \code{"X"}-\code{"XXXXX"} field
patterns except that the ISO 8601 UTC indicator \code{Z} \emph{will not} be used when
the local time offset is \code{0}.\tabular{lll}{
   Field Patterns \tab Output \tab Notes \cr
   \code{"x"} \tab \code{"-07"} \tab ISO 8601 basic format (h, optional m) \cr
   \code{"xx"} \tab \code{"-0700"} \tab ISO 8601 basic format (h & m) \cr
   \code{"xxx"} \tab \code{"-07:00"} \tab ISO 8601 extended format (h & m) \cr
   \code{"xxxx"} \tab \code{"-0700"} \tab ISO 8601 basic format (h & m, optional s) \cr
   \code{"xxxxx"} \tab \code{"-07:00"} \tab ISO 8601 extended format (h & m, optional s) \cr
}

}

}
}

\section{Examples}{

\subsection{Basics with \code{input} datetimes, formatting strings, and localization}{

With an input datetime of \code{"2018-07-04 22:05"} supplied as a string, we can
format to get just a date with the full year first, the month abbreviation
second, and the day of the month last (separated by hyphens):

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2018-07-04 22:05",
  format = "y-MMM-dd"
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "2018-Jul-04"
}\if{html}{\out{</div>}}

There are sometimes many options for each time part. Instead of using
\code{"y-MMM-dd"}, let's try a variation on that with \code{"yy-MMMM-d"}:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2018-07-04 22:05",
  format = "yy-MMMM-d"
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "18-July-4"
}\if{html}{\out{</div>}}

The output is localizable and so elements will be translated when supplying
the appropriate locale code. Let's use \code{locale = es} to get the month written
in Spanish:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2018-07-04 22:05",
  format = "yy-MMMM-d",
  locale = "es"
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "18-julio-4"
}\if{html}{\out{</div>}}

\code{POSIXct} or \code{POSIXlt} datetimes can serve as an \code{input} to \code{fdt()}. Let's
create a single datetime value where the timezone is set as \code{"Asia/Tokyo"}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = lubridate::ymd_hms("2020-03-15 19:09:12", tz = "Asia/Tokyo"),
  format = "EEEE, MMMM d, y 'at' h:mm:ss B (VVVV)"
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "Sunday, March 15, 2020 at 7:09:12 in the evening (Tokyo Time)"
}\if{html}{\out{</div>}}

If you're going minimal, it's possible to supply an input datetime string
without a \code{format} directive. What this gives us is formatted datetime
output that conforms to ISO 8601. Note that the implied time zone is UTC.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(input = "2018-07-04 22:05")
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "2018-07-04T22:05:00Z"
}\if{html}{\out{</div>}}
}

\subsection{Using locales stored in the \link{fdt_locales_lst} list}{

The \link{fdt_locales_lst} object is provided in \strong{bigD} to make it easier to
choose one of supported locales. You can avoid typing errors and every
element of the list is meant to work. For example, we can use the \code{"it"}
locale by accessing it from \link{fdt_locales_lst} (autocomplete makes this
relatively simple).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2018-07-04 22:05",
  format = "yy-MMMM-d",
  locale = fdt_locales_lst$it
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "18-luglio-4"
}\if{html}{\out{</div>}}
}

\subsection{Omission of date or time in \code{input}}{

You don't have to supply a full datetime to \code{input}. Just supplying the date
portion implies midnight (and is just fine if you're only going to present
the date anyway).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(input = "2018-07-04")
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "2018-07-04T00:00:00Z"
}\if{html}{\out{</div>}}

If you omit the date and just supply a time, \code{fdt()} will correctly parse
this. The current date on the user system will be used because we need to
create some sort of datetime value internally. Again, this is alright if
you just intend to present a formatted time value.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(input = "22:05")
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "2022-08-16T22:05:00Z"
}\if{html}{\out{</div>}}

To see all of the supported locales, we can look at the vector supplied by
the \code{\link[=fdt_locales_vec]{fdt_locales_vec()}} function.
}

\subsection{Using standardized forms with the \verb{standard_*()} helper functions}{

With an input datetime of \code{"2018-07-04 22:05(America/Vancouver)"}, we can
format the date and time in a standardized way with \code{standard_date_time()}
providing the correct formatting string. This function is invoked in the
\code{format} argument of \code{fdt()}:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2018-07-04 22:05(America/Vancouver)",
  format = standard_date_time(type = "full")
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "Wednesday, July 4, 2018 at 10:05:00 PM Pacific Daylight Time"
}\if{html}{\out{</div>}}

The locale can be changed and we don't have to worry about the particulars
of the formatting string (they are standardized across locales).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2018-07-04 22:05(America/Vancouver)",
  format = standard_date_time(type = "full"),
  locale = fdt_locales_lst$nl
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "woensdag 4 juli 2018 om 22:05:00 Pacific-zomertijd"
}\if{html}{\out{</div>}}

We can use different \code{type} values to control the output datetime string. The
default is \code{"short"}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2018-07-04 22:05(America/Vancouver)",
  format = standard_date_time()
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "7/4/18, 10:05 PM"
}\if{html}{\out{</div>}}

After that, it's \code{"medium"}:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2018-07-04 22:05(America/Vancouver)",
  format = standard_date_time(type = "medium")
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "Jul 4, 2018, 10:05:00 PM"
}\if{html}{\out{</div>}}

The \code{"short"} and \code{"medium"} types don't display time zone information in the
output. Beginning with \code{"long"}, the tz is shown.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2018-07-04 22:05(America/Vancouver)",
  format = standard_date_time(type = "long")
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "July 4, 2018 at 10:05:00 PM PDT"
}\if{html}{\out{</div>}}

If you don't include time zone information in the input, the \code{"UTC"} time
zone will be assumed:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2018-07-04 22:05",
  format = standard_date_time(type = "full")
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "Wednesday, July 4, 2018 at 10:05:00 PM GMT+00:00"
}\if{html}{\out{</div>}}
}

\subsection{Using flexible date and time (12- and 24-hour) formatting}{

The \strong{bigD} package supplies a set of lists to allow for flexible date and
time formatting (\link{flex_d_lst}, \link{flex_t24_lst}, and \link{flex_t12_lst}). These
are useful when you need a particular format that works across all locales.
Here's an example that uses the \code{"yMMMEd"} flexible date type by accessing it
from the \code{flex_d_lst} object, yielding a formatted date.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2021-01-09 16:32(America/Toronto)",
  format = flex_d_lst$yMMMEd,
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "Sat, Jan 9, 2021"
}\if{html}{\out{</div>}}

If we wanted this in a different locale, the locale-specific \code{format} pattern
behind the flexible date identifier would ensure consistency while moving to
that locale.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2021-01-09 16:32(America/Toronto)",
  format = flex_d_lst$yMMMEd,
  locale = "fr_CA"
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "sam. 9 janv. 2021"
}\if{html}{\out{</div>}}

Formatting as a 12-hour time with the \link{flex_t12_lst} list and using the
\code{"hms"} flexible type:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2021-01-09 16:32(America/Toronto)",
  format = flex_t12_lst$hms
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "4:32:00 PM"
}\if{html}{\out{</div>}}

The 24-hour variant, \link{flex_t24_lst}, has a similar \code{"Hms"} flexible type that
will give us a 24-hour version of the same clock time:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2021-01-09 16:32(America/Toronto)",
  format = flex_t24_lst$Hms
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> [1] "16:32:00"
}\if{html}{\out{</div>}}

A flexible date and time can be used together by enveloping the two in a
list (\strong{bigD} will handle putting the date and time together in a sensible
manner).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fdt(
  input = "2021-01-09 16:32(America/Toronto)",
  format = list(flex_d_lst$yMMMEd, flex_t24_lst$Hmv)
)
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{#> "Sat, Jan 9, 2021, 16:32 ET"
}\if{html}{\out{</div>}}
}
}

