% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ards.R
\name{restore_ards}
\alias{restore_ards}
\title{Restore an ARDS dataset}
\usage{
restore_ards(data, init_vars = FALSE, anal_var = "anal_var")
}
\arguments{
\item{data}{The input dataset to restore.  The input dataset should correspond
to the CDISC ARDS structure, such as that created by \code{get_ards}. However,
not all variables are required.  The only required variables are "anal_var", 
"statname", and "statval".  All other variables will be processed if available,
and ignored otherwise.}

\item{init_vars}{Whether or not to keep the initialization variables on
the restored data frames.  Default is FALSE. The initialization variables
include "studyid", "tableid", "adsns", "population", "time", and "where".
To keep these variables on the restored data frames, set \code{init_vars} 
to TRUE.}

\item{anal_var}{The name to use for the analysis variable column. This column 
is retained to positively identify the data frame. The default name is
is "anal_var". If you need a different name for this column, pass the name
as a quoted string.  To eliminate the column entirely, pass a NULL value
to this parameter.}
}
\value{
A list of data frames, transposed back into wide format.  The 
list will have one or more items, distinguished by the analysis variable.
The list item name will correspond to the name of the analysis variable.
}
\description{
The \code{restore_ards} function restores the ARDS data to wide format.
The wide data can then be used for reporting.
}
\details{
The \code{\link{init_ards}}, \code{\link{add_ards}}, and \code{\link{get_ards}} 
functions take data in wide format, and convert it to narrow format.  The
\code{restore_ards} function converts the narrow ARDS data back to wide format.

Wide format means there is a column for each statistic.  Narrow format
means all statistics are in a single column. 

Because each analysis variable can have any number of statistics, when
converting from narrow to wide, the resulting data frames can have different
numbers of colums and different column names.  Therefore, the \code{restore_ards}
function returns a list of data frames, one for each analysis variable.

For each data frame, the statistics will each be in a separate column, 
named according to the original statistic variable name.  The label of the 
statistics columns will be any value passed to the statistic description
("statdesc") for that analysis variable.

By default, the columns populated by \code{\link{init_ards}} will not be 
returned. These columns can be returned by setting the "init_vars" parameter
to TRUE. 

Once the ARDS data is restored and the statistics are back in separate
columns, it will be easy to create a report, figure, or other output.
}
\examples{
library(ards)
library(dplyr)

# Initialize the ARDS
# - These values will be common through the dataset
init_ards(studyid = "IRIS",
          tableid = "01", adsns = "iris",
          population = "all flowers",
          time = "1973")

# Perform analysis on Petal.Length
# - Using Species as a by-group
analdf1 <- iris |> 
  select(Petal.Length, Species) |> 
  group_by(Species) |> 
  summarize(n = n(),
            mean = mean(Petal.Length),
            std = sd(Petal.Length),
            min = min(Petal.Length),
            max = max(Petal.Length)) |> 
  add_ards(statvars = c("n", "mean", "std", "min", "max"),
           statdesc = c("Count", "Mean", "STD", "Minimum", "Maximum"),
           anal_var = "Petal.Length", trtvar = "Species")
           
# Perform analysis on Petal.Width
# - Using Species as a by-group
analdf2 <- iris |> 
  select(Petal.Width, Species) |> 
  group_by(Species) |> 
  summarize(n = n(),
            mean = mean(Petal.Width),
            std = sd(Petal.Width),
            min = min(Petal.Width),
            max = max(Petal.Width)) |> 
  add_ards(statvars = c("n", "mean", "std", "min", "max"),
           statdesc = c("Count", "Mean", "STD", "Minimum", "Maximum"),
           anal_var = "Petal.Width", trtvar = "Species")

# Get the ARDS
ards <- get_ards() 

# Convert back to wide format
res <- restore_ards(ards)

# View list names
print(names(res))
# [1] "Petal.Length" "Petal.Width" 

# Pull out Petal.Length
r1 <- res$Petal.Length

# View column names on Petal.Length
print(names(r1))
# [1] "Species"  "anal_var" "n"        "mean"     "std"      "min"      "max"    

# View stat data on Petal.Length
print(r1)
#      Species     anal_var  n  mean       std min max
# 1     setosa Petal.Length 50 1.462 0.1736640 1.0 1.9
# 2 versicolor Petal.Length 50 4.260 0.4699110 3.0 5.1
# 3  virginica Petal.Length 50 5.552 0.5518947 4.5 6.9

# Uncomment to view restored datasets
# View(res$Petal.Length)
# View(res$Petal.Width)

}
\seealso{
Other ards: 
\code{\link{add_ards}()},
\code{\link{get_ards}()},
\code{\link{init_ards}()}
}
\concept{ards}
