% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/GLMC.R
\name{buhlmannStraubTweedie}
\alias{buhlmannStraubTweedie}
\title{Combining the Buhlmann-Straub credibility model with a Tweedie GLM (Ohlsson, 2008)}
\usage{
buhlmannStraubTweedie(
  formula,
  data,
  weights,
  muHatGLM = FALSE,
  epsilon = 1e-04,
  maxiter = 500,
  verbose = FALSE,
  returnData = TRUE,
  cpglmControl = list(bound.p = c(1.01, 1.99)),
  balanceProperty = TRUE,
  optimizer = "bobyqa",
  y = TRUE,
  ...
)
}
\arguments{
\item{formula}{object of type \code{\link{formula}} that specifies which model should be fitted. Syntax follows
\code{\link[lme4]{lmer}}: e.g., \code{Y ~ x1 + x2 + (1 | Cluster)}. Only one random effect is allowed.}

\item{data}{an object that is coercible by \code{\link[data.table]{as.data.table}}, containing the variables in the model.}

\item{weights}{variable name of the exposure weight.}

\item{muHatGLM}{indicates which estimate has to be used in the algorithm for the intercept term. Default is \code{TRUE},
which uses the intercept as estimated by the GLM. If \code{FALSE}, the estimate of the Buhlmann-Straub credibility model is used.}

\item{epsilon}{positive convergence tolerance \eqn{\epsilon}; the iterations converge when
\eqn{||\theta[k] - \theta[k - 1]||^2_2/||\theta[k - 1]||^2_2 < \epsilon}. Here, \eqn{\theta[k]} is the parameter vector at the \eqn{k^{th}} iteration.}

\item{maxiter}{maximum number of iterations.}

\item{verbose}{logical indicating if output should be produced during the algorithm.}

\item{returnData}{logical indicating if input data has to be returned.}

\item{cpglmControl}{a list of parameters to control the fitting process in the GLM part. By default,
\code{cpglmControl = list(bound.p = c(1.01, 1.99))} which restricts the range of the power parameter p to [1.01, 1.99] in the fitting
process. This list is passed to \code{\link[cplm]{cpglm}}.}

\item{balanceProperty}{logical indicating if the balance property should be satisfied.}

\item{optimizer}{a character string that determines which optimization routine is to be used in estimating the index and the dispersion parameters.
Possible choices are \code{"nlminb"} (the default, see \code{\link[stats]{nlminb}}), \code{"bobyqa"} (\code{\link[minqa]{bobyqa}}) and \code{"L-BFGS-B"} (\code{\link[stats]{optim}}).}

\item{y}{logical indicating whether the response vector should be returned as a component of the returned value.}

\item{...}{arguments passed to \code{\link[cplm]{cpglm}}.}
}
\value{
An object of type \code{buhlmannStraubTweedie} with the following slots:

\item{call}{the matched call}

\item{CredibilityResults}{results of the Buhlmann-Straub credibility model.}

\item{fitGLM}{the results from fitting the GLM part.}

\item{iter}{total number of iterations.}

\item{Converged}{logical indicating whether the algorithm converged.}

\item{LevelsCov}{object that summarizes the unique levels of each of the contract-specific covariates.}

\item{fitted.values}{the fitted mean values, resulting from the model fit.}

\item{prior.weights}{the weights (exposure) initially supplied.}

\item{y}{if requested, the response vector. Default is \code{TRUE}.}
}
\description{
Fit a single-level random effects model using Ohlsson's methodology combined with Buhlmann-Straub credibility.
This function estimates the power parameter p. For fixed p, see \code{\link{buhlmannStraubGLM}}.
}
\details{
When estimating the GLM part, this function uses the \code{\link[cplm]{cpglm}} function from the \code{cplm} package.
}
\examples{
\donttest{
data("hachemeister", package = "actuar")
X = as.data.frame(hachemeister)
Df = reshape(X, idvar = "state",
             varying = list(paste0("ratio.", 1:12), paste0("weight.", 1:12)),
             direction = "long")
Df$time_factor = factor(Df$time)
fit = buhlmannStraubTweedie(ratio.1 ~ time_factor + (1 | state), Df,
                            weights = weight.1, epsilon = 1e-6)
summary(fit)
ranef(fit)
fixef(fit)
}
}
\references{
Campo, B.D.C. and Antonio, Katrien (2023). Insurance pricing with hierarchically structured data an illustration with a workers' compensation insurance portfolio. \emph{Scandinavian Actuarial Journal}, doi: 10.1080/03461238.2022.2161413

Ohlsson, E. (2008). Combining generalized linear models and credibility models in practice. \emph{Scandinavian Actuarial Journal} \bold{2008}(4), 301–314.
}
\seealso{
\code{\link{buhlmannStraubTweedie-class}}, \code{\link{buhlmannStraub}}, \code{\link{buhlmannStraubGLM}},
\code{\link{hierCredTweedie}}, \code{\link[cplm]{cpglm}}, \code{\link{plotRE}}, \code{\link{adjustIntercept}}, \code{\link{BalanceProperty}}
}
