% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/abms.R
\name{gibbs_abms}
\alias{gibbs_abms}
\title{Bayesian variable selection models via a spike-and-slab methodology.}
\usage{
gibbs_abms(
  y,
  Covariates,
  family = "LiR",
  first_excluded = 0,
  nchain = 10000,
  burnin = 2000,
  tau2 = 1000,
  rho = 1,
  ni = rep(1, length(y)),
  alpha = 0.5,
  a0 = 1,
  b0 = 1,
  d = 2,
  b2 = 1/2,
  model_fixed = NULL,
  WomackPrior = TRUE,
  a_bb = 1,
  b_bb = 1,
  count.iteration = TRUE
)
}
\arguments{
\item{y}{A vector of size \eqn{n} with observed responses. It can also be a (\eqn{n x 1}) matrix.}

\item{Covariates}{A data.frame object with the predictors (without the intercept) for which we want to test if they are relevant to the response variable. It can also be a (\eqn{n x p}) matrix.}

\item{family}{A character object that describes the hierarchical regression model that will be used.
If \code{family="LiR"}, then a Linear regresion model will be fitted (gaussian errors).
If \code{family="LoR"}, then a Logistic regresion model will be fitted (binomial distribution).
If \code{family="NBR"}, then a Negative Binomial regresion model will be fitted (mean \eqn{r(1-p)/p}).
If \code{family="QR"}, then a Quantile regresion model will be fitted (Asymmetric Laplace errors).
If \code{family="SNR"}, then a Skew normal regresion model will be fitted (Skew-Normal errors).
The argument is fixed at \code{family="LiR"} by default.}

\item{first_excluded}{A non-negative integer that indicates which first columns will not be tested. For example, if \code{first_excluded=2}, the two first columns of \code{Covariates} will not be tested. Intercept is always excluded for the selection process.}

\item{nchain}{The Gibbs sampler's chain size, it must be a non-negative integer. The default value is 10,000}

\item{burnin}{The burn-in period of the Gibbs sampler, it must be a non-negative integer and greater than \code{nchain}. The default value is 2,000}

\item{tau2}{The variance prior of each coefficient, it must be a positive real number. Fixed at 1 by deafault}

\item{rho}{The parameter of the Womack prior, it must be a positive real number. Fixed at 1 by deafault}

\item{ni}{For Logistic regression only. A vector of size \eqn{n} that represent the i-th individual size (the size parameter of the Binomial distribution) that it must be a positive integer. It can also be a (\eqn{n x 1}) matrix. For default, all individual size are fixed at 1.}

\item{alpha}{For Quantile regression only. The desired quantile for which we want to perform Quantile regression. \code{alpha} must be between (\eqn{0,1}). By default, \code{alpha=0.5}, that is, median regression.}

\item{a0}{This argument depends on the family choosen.
For \code{family="LiR"}, is the shape hyper-parameter of the \eqn{Gamma} prior to the variance parameter (\eqn{\sigma^2}) of the Gaussian distribution.
For \code{family="NBR"} is the shape hyper-parameter of the \eqn{Gamma} prior to the parameter \eqn{r}  the Negative Binomial distribution (the number of successes until the experiment is stopped).
For \code{family="QR"} is the shape hyper-parameter of the \eqn{Gamma} prior to thevariance parameter (\eqn{\sigma^2}) of the Asymmetric Laplace distribution.
Note thas this argument do not exist for \code{family=LoR} and \code{family=SNR}. For all hierarchical regression models, it must be a positive real number and its fixed at 1 by deafault.}

\item{b0}{This argument depends on the family choosen.
For \code{family="LiR"} is the scale hyper-parameter of the \eqn{Gamma} prior to the variance parameter (\eqn{\sigma^2}) of the Gaussian distribution.
For \code{family="NBR"} is the scale hyper-parameter of the \eqn{Gamma} prior to the parameter \eqn{r}  the Negative Binomial distribution (the number of successes until the experiment is stopped).
For \code{family="QR"} is the scale hyper-parameter of the \eqn{Gamma} prior to the variance parameter (\eqn{\sigma^2}) of the Asymmetric Laplace distribution.
Note thas this argument do not exist for \code{family=LoR} and \code{family=SNR}. For all hierarchical regression models, it must be a positive real number and its fixed at 1 by deafault.}

\item{d}{For the Skew-Normal regression only. It is the location hyper-parameter of the t-student prior to the parameter \eqn{lambda} (asymmetric parameter of the Skew-Normal distribution). By default is fixed at 2, which is recommended.}

\item{b2}{For the Skew-Normal regression only. It is the scale hyper-parameter of the t-student prior to the parameter lambda (asymmetric parameter of the Skew-Normal distribution). By default is fixed at 1/2, which is recommended.}

\item{model_fixed}{Either \code{NULL} or a vector that indicates which model will be fixed to perform parameter estimation only under such a model. For example, if there are only three predictors and \code{model.fixed=c(1,3)}, only parameter estimation will be performed where only the first and third predictors are included. If \code{NULL}, model selection will also be performed. Fixed at \code{NULL} by default.}

\item{WomackPrior}{A logical argument. If \code{TRUE}, the Womack prior for the model space will be used. Otherwise, the Beta-Binomial prior with shape parameters \code{a_bb} and \code{b_bb} will be used. Fixed at \code{TRUE} by default}

\item{a_bb}{A numeric vector of length 1. The first shape parameter of the Beta-Binomial prior. Recomended value is \code{a_bb=1}.}

\item{b_bb}{A numeric vector of length 1. The second shape parameter of the Beta-Binomial prior. Recomended value is \code{b_bb=p_selection^u}, where \code{u}>1, and \code{p_selection} is the number of predictors under the selection process.}

\item{count.iteration}{A logical argument. If \code{TRUE}, a counter for the Gibbs sampler iterations will be displayed. Fixed at \code{TRUE} by deafult.}
}
\value{
A abms object with the following variables:

\item{family}{This character object prints the name of the fitted hierarchical regression model. It needs to be extracted from the list 'Default'.}

\item{prednames}{A character object that prints the predictors names, using the columns names of the \code{Covariates} argument. It needs to be extracted from the list 'Default'.}

\item{Seconds}{How many seconds the method took. It needs to be extracted from the list 'Default'.}

\item{tau2}{The \code{tau2} that was used as argument.}

\item{y}{The \code{y} response vector that was used as argument.}

\item{Covariates}{The \code{Covariates} data frame or matrix that was used as argument.}

\item{beta_chain}{The coefficients sample for each Gibbs sampler iteration. A (\code{nchain} x \eqn{p}) matrix}

\item{sigma2_chain}{For the Linear, Quantile and Skew-Normal regression only. The variance parameter (\eqn{\sigma^2}) sample for each Gibbs sampler iteration. A (\code{nchain} x 1) matrix}

\item{r_chain}{For the Negative-Binomial regression only. The number of failure parameter (\eqn{r}) sample for each Gibbs sampler iteration. A (\code{nchain} x 1) matrix}

\item{lambda_chain}{For the Skew-Normal regression only. The asymmetric parameter (\eqn{\lambda}) sample for each Gibbs sampler iteration. A (\code{nchain} x 1) matrix}

\item{model_chain}{The model selected at each Gibbs sampler iteration. A (\code{nchain} x \eqn{p}) matrix.}

\item{Z_chain}{For internal use.}

\item{t_chain}{For internal use.}
}
\description{
A Bayesian model selection methodology based on the spike-and-slab strategy and an augmentation technique for Linear, Logistic, Negative Binomial, Quantile, and Skew Normal Regression.
The model considers a response vector \eqn{y} of size \eqn{n} and \eqn{p} predictors to perform coefficient estimation and asses which ones are relevant to explain the response distribution. Other parameters related to the family selected are also estimated.
Summary results can be provided using the \code{summary_gibbs()} R function.
}
\examples{
##################################################
## 	    	Gibbs for Linear Regression 		##
##################################################

## Simulating data
set.seed(31415)
N<-200
r_beta<-as.matrix(c(1, 0, 2, 0))
r_p<-length(r_beta)
r_sigma2<-1.5
X<-matrix( c(rep(1, N), rnorm((r_p -1)*N)), ncol=r_p )
Xbeta<-X\%*\%r_beta
y<-rnorm(N, mean=Xbeta , sd=sqrt(r_sigma2))
Covariates<-X[,2:(length(r_beta))];
colnames(Covariates)<-c("X1", "X2", "X3")

## Fitting the model
fit<- gibbs_abms(y, Covariates, family="LiR", first_excluded=0, nchain=1000, burnin=20,
 a0=1, b0=1)

summary_gibbs(fit, BF=FALSE)	#Summary results

## For more examples, see "Model Ilustrations.R" file in
## https://github.com/SirCornflake/BMS
}
\references{
{Azzalini (1985). A class of distributions which includes the normal ones, Scandinavian Journal of Statistics 12(2): 171:178.}

{Bayes, C. and Branco, M. (2007). Bayesian inference for the skewness parameter of the scalar skew-normal distribution. Brazilian Journal of Probability and Statistics. 21: 141:163.}

{Kotz, S., Kozubowski, T. and Podgorski, K. (2001). The Laplace Distribution and Generalization, first edn, Birkhauser Basel.}

{Polson, N., Scott, J., and Windle, J. (2013). Bayesian Inference for Logistic Models Using Polya Gamma Latent Variables. Journal of the American Statistical Association, 108: 1339:1349.}

{Zhou, W. and Carin, L. (2013). Negative Binomial Process Count and Mixture Modeling. arXiv:1405.0506v1.}
}
