/*
  Author: Xuye Luo
  Date: December 12, 2025
*/

#include <Rcpp.h>
#include <cmath>
using namespace Rcpp;

// [[Rcpp::interfaces(r, cpp)]]

/* 
//' @title Calculate Upsilon Statistic for Matrix (C++ Backend)
//' @description Calculates the Upsilon association statistic for a contingency table.
//' @param x NumericMatrix representing the contingency table.
//' @return The calculated statistic.
 */

// [[Rcpp::export]]
double upsilon_statistic_cpp(const NumericMatrix &x) {
  
  double N = sum(x);
  
  // Handle empty table case
  if (N == 0) {
    return 0.0;
  }

  int nr = x.nrow();
  int nc = x.ncol();
  
  // Calculate marginal sums
  NumericVector row_sum = rowSums(x);
  NumericVector col_sum = colSums(x);
  
  double sum_sq_diff = 0.0;
  double inv_N = 1.0 / N; 

  for (int j = 0; j < nc; ++j) {
    double c_sum = col_sum[j];
    
    if (c_sum == 0) continue;

    for (int i = 0; i < nr; ++i) {
      double r_sum = row_sum[i];
      
      // Skip row if marginal sum is 0
      if (r_sum == 0) continue;
      
      double O = x(i, j);
      double E = r_sum * c_sum * inv_N;
      
      double diff = O - E;
      sum_sq_diff += diff * diff; 
    }
  }

  // Formula: Upsilon = Sum((O-E)^2) / (N / (nr*nc))
  //                 = Sum((O-E)^2) * (nr*nc) / N
  double statistic = sum_sq_diff * (double)nr * (double)nc * inv_N;
  
  if (statistic < 0) statistic = 0;

  return statistic;
}
