#' Solve the sign-aligned frequency-severity (SAFE) model
#'
#' This function fits a Poisson-Gamma regression framework that aligns the signs of certain predictors in both
#' the Poisson frequency component and the Gamma severity component. The solution path is computed at a
#' sequence of tuning parameter values (\code{lambda}).
#'
#' @param x A numeric matrix of dimensions \eqn{n \times p}, where \eqn{n} is the number of observations
#'   and \eqn{p} is the number of predictors.
#' @param y A numeric vector of length \eqn{n}, representing the loss values (severity). These are assumed
#'   to follow a Gamma distribution with shape parameter \code{alpha} and scale parameter
#'   \eqn{\theta = \exp(x^\top \gamma) / \alpha}.
#' @param k A numeric vector of length \eqn{n}, representing the number of claims (frequency), assumed to follow
#'   a Poisson distribution with mean \eqn{\mu = x^\top \beta}.
#' @param lambda A user-supplied numeric vector of tuning parameters. The function will compute the solution
#'   for each value in \code{lambda}.
#' @param alpha The shape parameter for the Gamma distribution (default is 1).
#' @param eps Convergence tolerance for updating \code{beta} (default is \code{1e-8}).
#' @param eps2 Convergence tolerance for updating \code{gamma} (default is \code{1e-8}).
#' @param maxit An integer specifying the maximum number of iterations (default is \code{1e5}).
#' @param ind_p A user-provided vector specifying which predictors should share the same sign
#'   across frequency and severity.
#' @param int_gam Optional numeric vector or matrix providing initial values for \code{gamma}; defaults to \code{NULL}.
#' @param int_beta Optional numeric vector or matrix providing initial values for \code{beta}; defaults to \code{NULL}.
#'
#' @details
#' The function uses an **accelerated proximal gradient descent** algorithm to simultaneously estimate
#' \code{beta} (for the Poisson frequency model) and \code{gamma} (for the Gamma severity model) under
#' a sign-alignment constraint for selected predictors (controlled by \code{ind_p}).
#'
#' @return An object of class \code{safe}, which is a list containing:
#' \item{\code{beta}}{A \eqn{p \times L} matrix of frequency-model coefficients, where \eqn{p} is the number of predictors
#'       and \eqn{L} is the number of \code{lambda} values.}
#' \item{\code{gamma}}{A \eqn{p \times L} matrix of severity-model coefficients, with the same dimensions
#'       as \code{beta}.}
#' \item{\code{lambda}}{The (sorted) sequence of \code{lambda} values used in the estimation.}
#' \item{\code{npass_beta}}{Total number of iterations used to update \code{beta} across all \code{lambda} values.}
#' \item{\code{npass_gamma}}{Total number of iterations used to update \code{gamma} across all \code{lambda} values.}
#' \item{\code{jerr}}{An integer flag for warnings or errors; \code{0} indicates no issues encountered.}
#'
#' @examples
#' set.seed(1)
#' n <- 100
#' p <- 5
#' x <- matrix(rnorm(n * p), nrow = n, ncol = p)
#' beta_true <- rep(0.1, 5)
#' gamma_true <- c(rep(1, 3), -1, -1)
#' mu <- x %*% beta_true
#' k <- rpois(n, lambda = exp(mu))
#' alpha_val <- 1
#' theta <- exp(x %*% gamma_true) / alpha_val
#' y <- rgamma(n, shape = alpha_val, scale = theta)
#' lambda_val <- 1
#' fit <- safe(x, y, k, 1, ind_p = c(1, 1, 1, 0, 0))
#'
#' @keywords model regression
#' @useDynLib SAFEPG, .registration=TRUE
#' @export


safe <- function(x, y, k, lambda, alpha = 1,
  eps = 1e-8, maxit = 1e+5, eps2 = 1e-8, ind_p, int_gam = NULL, 
  int_beta = NULL) {
  ####################################################################
  y <- drop(y)
  x <- as.matrix(x)
  np <- dim(x)
  nobs <- as.integer(np[1])
  nvars <- as.integer(np[2])
  if (length(y) != nobs) 
    stop("x and y have different number of observations")
  ####################################################################
  if(is.null(int_beta) || is.null(int_gam)){
    fit1 <- ini.safe(x, y, k)
    int_beta <- fit1$beta
    int_gam <- fit1$gamma
  }

  maxit <- as.integer(maxit)
  eps <- as.double(eps)
  eps2 <- as.double(eps2)
  alpha <- as.double(alpha)
  
  ulam <- as.double(rev(sort(lambda)))
  nlam <- as.integer(length(lambda))
  gamma <- matrix(0, nvars, nlam)
  beta <- matrix(0, nvars, nlam)
  npass_beta <- rep(0, nlam)
  npass_gamma <- rep(0, nlam)
  jerr <- as.integer(0)
  ####################################################################
  fit <- .Fortran("safe_fista", as.double(x), as.double(y), as.integer(k), 
    nobs, nvars, nlam, ulam, eps, eps2, gamma = as.double(gamma), 
    beta = as.double(beta), alpha = alpha, maxit, npass_beta = as.integer(npass_beta), 
    npass_gamma = as.integer(npass_gamma), 
    int_beta = as.double(int_beta), int_gam = as.double(int_gam), as.integer(ind_p),
    jerr = jerr, PACKAGE="SAFEPG")
  gamma <- matrix(fit$gamma[seq(nvars * nlam)], nvars, nlam)
  beta <- matrix(fit$beta[seq(nvars * nlam)], nvars, nlam)
  outlist <- list(beta = beta, gamma = gamma, npass_beta = fit$npass_beta, lambda = ulam,
    npass_gamma = fit$npass_gamma, jerr = fit$jerr)
  class(outlist) <- c("safe")
  outlist
}  


ini.safe <- function(x, y, k, alpha = 1, 
  eps = 1e-8, maxit = 1e+5, eps2 = 1e-8) {
  ####################################################################
  y <- drop(y)
  x <- as.matrix(x)
  np <- dim(x)
  nobs <- as.integer(np[1])
  nvars <- as.integer(np[2])
  if (length(y) != nobs) 
    stop("x and y have different number of observations")
  ####################################################################

  ind_p=rep(0, nvars)
  int_beta=rep(0,nvars)
  int_gam=rep(0,nvars)
  lambda = 0
  maxit <- as.integer(maxit)
  eps <- as.double(eps)
  eps2 <- as.double(eps2)
  alpha <- as.double(alpha)
  ulam <- as.double(rev(sort(lambda)))
  nlam <- as.integer(length(lambda))
  gamma <- as.double(matrix(0, nvars, nlam))
  beta <- as.double(matrix(0, nvars, nlam))
  npass_beta <- integer(nlam)
  npass_gamma <- integer(nlam)
  jerr <- integer(1)
  ####################################################################
  fit <- .Fortran("safe_fista", as.double(x), as.double(y), as.integer(k), 
    nobs, nvars, nlam, ulam, eps, eps2, gamma = as.double(gamma), 
    beta = as.double(beta), alpha = alpha, maxit, npass_beta = as.integer(npass_beta), 
    npass_gamma = as.integer(npass_gamma), 
    int_beta = as.double(int_beta), int_gam = as.double(int_gam), as.integer(ind_p),
    jerr = jerr, PACKAGE="SAFEPG")
  gamma <- matrix(fit$gamma[seq(nvars * nlam)], nvars, nlam)
  beta <- matrix(fit$beta[seq(nvars * nlam)], nvars, nlam)
  return(list(beta = beta, gamma = gamma, npass_beta = fit$npass_beta, 
    npass_gamma = fit$npass_gamma, jerr = fit$jerr))
} 