#' Count skeleton root pixels with certain widths in the base root image
#'
#' \code{skelPxWidth} - Counts the skeleton pixels that belong to root pieces
#' of certain (circular) widths (diameter <3px, 3-7px, >7px).\cr
#' For each white root pixel in the skeleton image, it checks how large (in
#' terms of the categories <3px, 3-7px, >7px, which equal a radius of <1px,
#' 1-3px, >=4px) the circle of white root pixels is that surrounds it in the
#' base root image. \cr
#' Optionally, a masking layer can be specified that indicates which pixels of
#' the skeleton image should be checked (the radius of the white circle is
#' still check in the complete base image).
#'
#' @param base_img Base image (provided by the RootDetector). Can be a PNG,
#' i.e., an array with 3 dimensions (3 layers each containing a 2-dim. numeric
#' matrix with values between 0 and 1), or a 2-dim. matrix.
#' @param skel_img Skeleton image with the same number of rows and columns
#' as \code{base_img} (provided by the RootDetector). Can be a PNG,
#' i.e., an array with 3 dimensions (3 layers each containing a 2-dim. numeric
#' matrix with values between 0 and 1), or a 2-dim. matrix.
#' @param mask 2-dim. true/false-matrix with the same number of rows and columns
#' as \code{base_img} (optional, default = NULL, interpreted as a matrix
#' consisting only of TRUEs, i.e., nothing is "removed" from the image).
#'
#' @return \code{skelPxWidth} Numeric vector of length 3 containing the counts.
#'
#' @export
#' @rdname skelPxWidth
#'
#' @examples
#' # In this example there are 2 white root pixels in the skeleton image.
#' # The left one is completely surrounded by white pixels in the base image,
#' # it falls into Category 2 (3-7px). The bottom right one has a black
#' # neighboring pixel and thus falls in to Category 1 (<3px). Thus, the result
#' # is Categorie 1: 1 pixel, Cat. 2: 1 pixel, Cat. 3: 0 pixels.
#' skelPxWidth(base_img = matrix(c(1,1,1,0,
#'                      1,1,1,0,
#'                      1,1,1,1), ncol = 4, nrow = 3, byrow = TRUE),
#'             skel_img = matrix(c(0,0,0,0,
#'                      0,1,0,0,
#'                      0,0,0,1), ncol = 4, nrow = 3, byrow = TRUE))
#' # Similar example with a mask which makes the function "ignore" the right
#' # side of the skeleton image.
#' # The function still identifies the left white pixel as of Category 2.
#' skelPxWidth(matrix(c(1,1,1,0,
#'                      1,1,1,0,
#'                      1,1,1,1), ncol = 4, nrow = 3, byrow = TRUE),
#'             matrix(c(0,0,0,0,
#'                      0,1,0,0,
#'                      0,0,0,1), ncol = 4, nrow = 3, byrow = TRUE),
#'              matrix(c(TRUE,TRUE,FALSE,FALSE,
#'                       TRUE,TRUE,FALSE,FALSE,
#'                       TRUE,TRUE,FALSE,FALSE), ncol = 4, nrow = 3,
#'                       byrow = TRUE))
skelPxWidth <- function(base_img, skel_img, mask = NULL){
  # Give the image three layers if it is only a matrix.
  if(length(dim(base_img))==2){
    base_img <- array(c(base_img, base_img, base_img),
                      dim = c(dim(base_img),3))
  }
  if(length(dim(skel_img))==2){
    skel_img <- array(c(skel_img, skel_img, skel_img),
                      dim = c(dim(skel_img),3))
  }
  # Set the result counter.
  valid_counts <- c(0,0,0)
  # Go through all pixels and check them.
  for(x in 1:nrow(skel_img[,,1])){
    for(y in 1:ncol(skel_img[,,1])){
      if(!is.null(mask) && !mask[x,y]){
        next
      }
      # If the pixel is white in the skeleton image
      if (sum(skel_img[x, y,] == 1)==3) {
        # Check if the pixel has a valid neighborhood in the base image
        white_px_radius <- 0
        for(test_radius in c(1,4)){
          test_neigh_coords <- getNeighborCoords(c(x,y), radius = test_radius,
                                                 dims_px = dim(base_img),
                                                 type = "area", tol = "strict",
                                                 tol_val = 0.5)
          # If there are any non-white (not 1) pixels in the neighborhood break
          # (not necessary to check a bigger radius),
          # else save it as a successful test radius.
          if(nrow(test_neigh_coords)==0 ||
             sum(c(base_img[,,1][test_neigh_coords],
                   base_img[,,2][test_neigh_coords],
                   base_img[,,3][test_neigh_coords])-1)<0){
            break()
          } else {
            white_px_radius <- test_radius
          }
        }
        if (white_px_radius == 0) {
          valid_counts[1] <- valid_counts[1] + 1
        } else if (white_px_radius == 1) {
          valid_counts[2] <- valid_counts[2] + 1
        } else if (white_px_radius == 4) {
          valid_counts[3] <- valid_counts[3] + 1
        } else {
          stop("This radius should not have occurred.")
        }
      }
    }
  }

  return(valid_counts)
}

