#' @title Plot Samples
#' @description This function creates graphs for the inputs of the RHC_function model.
#' @details
#' The function takes input data and specific row indices, creates sample charts, and returns the plots.
#' @param final_data_st A data frame containing standardized data from the first function.
#' @param row_indices A vector of row indices specifying which rows to use for creating sample plots.
#' @param plot_title_prefix A prefix for the plot titles (default is "Sample").
#' @param ncol Number of columns for arranging the plots. Default is 3.
#' @return A list of sample charts.
#' @examples
#' data(canopy_oc_file)
#' data(trait_file)
#' final_data_st <- prepare_RHC_data(canopy_oc_file, trait_file)
#' row_indices <- 1:17
#' plots.samples <- plot_samples(final_data_st, row_indices, plot_title_prefix = "Sample", ncol = 3)
#' @importFrom ggplot2 ggplot
#' @importFrom ggplot2 aes
#' @importFrom ggplot2 geom_bar
#' @importFrom ggplot2 labs
#' @importFrom ggplot2 theme_minimal
#' @importFrom ggplot2 theme
#' @importFrom ggplot2 element_text
#' @importFrom ggplot2 margin
#' @importFrom ggplot2 element_blank
#' @importFrom ggplot2 element_line
#' @importFrom ggplot2 geom_text
#' @importFrom ggplot2 coord_cartesian
#' @importFrom ggplot2 guides
#' @importFrom ggplot2 guide_legend
#' @importFrom ggplot2 expand_limits
#' @importFrom gridExtra grid.arrange
#' @name plot_samples
#' @export
utils::globalVariables(c("Score", "Variable"))

plot_samples <- function(final_data_st, row_indices, plot_title_prefix = "Sample", ncol = 3) {
  plots.samples <- list()
  colors <- grDevices::rainbow(length(row_indices))

  for (i in seq_along(row_indices)) {
    row_index <- row_indices[i]
    sample_data <- final_data_st[row_index, c(1:15)]
    sample_data <- as.numeric(sample_data)

    sample_df <- data.frame(
      Variable = colnames(final_data_st),
      Score = sample_data,
      Source = paste(plot_title_prefix, i, "(Standardized)")
    )

    mean_score <- mean(sample_data)

    plot <- ggplot(sample_df, aes(x = Score, y = Variable)) +
      geom_bar(stat = "identity", fill = colors[i], width = 0.7) +
      labs(
        x = "Standardized Scores", y = "Model Input Variables",
        title = paste(plot_title_prefix, row_index, ": ", sprintf("%.2f", mean_score))
      ) +
      theme_minimal() +
      theme(
        axis.text.y = element_text(size = 10, hjust = 1, vjust = 0.1, color = "black"),
        axis.text.x = element_text(size = 10, color = "black"),
        axis.title.x = element_text(margin = margin(t = 5), face = "bold"),
        axis.title.y = element_text(face = "bold"),
        panel.grid = element_blank(),
        axis.line = element_line(color = "gray"),
        plot.title = element_text(hjust = 0.5, vjust = 0, face = "bold", size = 14)
      ) +
      geom_text(aes(label = sprintf("%.2f", Score)), vjust = 0.5, color = "black", size = 4, hjust = -0.1) +
      coord_cartesian(xlim = c(0, 1)) +
      guides(fill = guide_legend(title = "Legend")) +
      expand_limits(x = max(sample_df$Score) * 1.2)

    plots.samples[[i]] <- plot
  }

  grid.arrange(grobs = plots.samples, ncol = ncol)
  return(plots.samples)
}
