% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rgn.R
\name{rgn}
\alias{rgn}
\title{Robust Gauss Newton optimization}
\usage{
rgn(
  simFunc,
  simTarget = 0,
  weights = NULL,
  par,
  lower,
  upper,
  control = NULL,
  ...
)
}
\arguments{
\item{simFunc}{is a function that simulates a (vector) response, with first argument the vector of parameters over which optimization is performed}

\item{simTarget}{is the target vector that \code{simFunc} is trying to match}

\item{weights}{is a vector of weights used in the WSS objective function. Defaults to equal weights.}

\item{par}{is the vector of initial parameters}

\item{lower}{is the lower bounds on parameters}

\item{upper}{is the upper bounds on parameters}

\item{control}{list of RGN settings
\itemize{
\item{\code{control$n.multi} is number of multi-starts
        (i.e. invocations of optimization with different initial parameter estimates). Default is 1.}
\item{\code{control$iterMax} is maximum iterations. Default is 100.}
\item{\code{control$dump} is level of diagnostic outputs between 0 (none) and 3 (highest). Default is 0.}
\item{\code{control$keep.multi} (TRUE/FALSE) controls whether diagnostic output from each multi-start is recorded. Default is FALSE.}
\item{\code{control$logFile} is log file name}
}}

\item{...}{other arguments to \code{simFunc()}}
}
\value{
List with
\itemize{
\item{\code{par}, the optimal parameters}
\item{\code{value}, the optimal objective function value}
\item{\code{sim}, the simulated vector using optimal parameters}
\item{\code{residuals}, the vector of residuals using optimal parameters}
\item{\code{counts}, the total number of function calls}
\item{\code{convergence}, an integer code indicating reason for completion.
\code{1} maximum iterations reached,
\code{2} relative reduction in function value small.
\code{3} absolute reduction in function value small
\code{4} relative change in parameters small}
}
}
\description{
\code{rgn} performs optimization of weighted-sum-of-squares (WSS) objective function using the Robust Gauss Newton algorithm
}
\details{
\code{rgn} minimizes the objective function \code{sum((weights*(simFunc-simTarget)^2))},
which is a sum of squared weighted residuals (\code{residuals=weights*(simFunc-simTarget)}).
Note \code{simFunc} corresponds to the vector of residuals when default
arguments for \code{simTarget} and \code{weights} are used.
}
\examples{
# Example 1: Rosenbrock
simFunc_rosenbrock=function(x) c(1.0-x[1],10.0*(x[2]-x[1]**2))
rgnOut = rgn(simFunc=simFunc_rosenbrock,
             par=c(-1.0,  0.0), lower=c(-1.5, -1.0), upper=c( 1.5,  3.0),
             simTarget=c(0,0))
rgnOut$par #optimal parameters
rgnOut$value #optimal objective function value

# Example 2: Hymod
\donttest{
data("BassRiver") # load Bass River hydrological data
rgnOut = rgn(simFunc=simFunc_hymod,
             par=c(400.,0.5,0.1,0.2,0.1),
             lower=c(1.,0.1,0.05,0.000001,0.000001),
             upper=c(1000.,2.,0.95,0.99999,0.99999),
             simTarget=BassRiverData$Runoff.mm.day[365:length(BassRiverData$Date)],
             stateVal=c(100.0,30.0,27.0,25.0,30.0,0.0,0.0,0.0), # initial states for hymod
             nWarmUp=365,                                       # warmup period
             rain=BassRiverData$Rain.mm,                        # precip input
             pet=BassRiverData$ET.mm)                           # PET input
rgnOut$par #optimal parameters
rgnOut$value #optimal objective function value
}

}
