#' Calculate intersection metrics in tracks
#'
#' \code{track_intersection()} calculates the number of unique intersections between trajectories.
#' The function also supports testing with simulations and different permutation procedures for the coordinates
#' of the simulated trajectories' origins to compute *p*-values. This allows for a robust assessment of the intersection metrics,
#' enabling users to evaluate the significance of the observed intersections in relation to simulated trajectories.
#'
#' @param data A \code{track} R object, which is a list consisting of two elements:
#'    * \strong{\code{Trajectories}}: A list of interpolated trajectories, where each trajectory is a series of midpoints between consecutive footprints.
#'    * \strong{\code{Footprints}}: A list of data frames containing footprint coordinates, metadata (e.g., image reference, ID), and a marker indicating whether the footprint is actual or inferred.
#' @param test Logical; if \code{TRUE}, the function compares the observed intersection metrics against. Default is \code{FALSE}.
#' @param H1 A character string specifying the alternative hypothesis to be tested. Options are \code{"Lower"} for testing whether the observed intersections are significantly lower than the simulated ones (e.g., coordinated or gregarious movement), or \code{"Higher"} for testing whether the observed intersections are significantly higher than the simulated ones (e.g., predatory or chasing events).
#' @param sim A \code{track simulation} R object consisting of a list of simulated trajectories to use for comparison when \code{test = TRUE}.
#' @param origin.permutation A character string specifying the method for permutation of the coordinates of the simulated trajectories' origins.
#' Options include \code{"None"}, \code{"Min.Box"}, \code{"Conv.Hull"}, or \code{"Custom"}. Default is \code{"None"}.
#' @param custom.coord A matrix of custom coordinates that define the vertices of an area for permutation of the coordinates of the simulated
#' trajectories' origins.
#'
#' @details
#' The \code{track_intersection()} function is designed to calculate the number of unique intersections between trajectories
#' and to evaluate their statistical significance through hypothesis testing based on simulated tracks. This process provides a
#' robust framework for comparing observed intersections against those expected under random conditions, allowing users to test
#' specific behavioral hypotheses related to the movement patterns of trackmakers.
#'
#' Hypothesis testing is controlled by the \code{H1} argument, which defines the \strong{alternative hypothesis} to be evaluated.
#' This argument is crucial for interpreting the statistical results, as it determines whether the function will test for
#' \strong{reduced} or \strong{increased} intersection counts compared to simulated trajectories.
#'
#' The \code{H1} argument accepts two possible values:
#' - \code{"Lower"}: This option tests whether the observed intersections are significantly lower than those generated by simulations.
#'   This scenario corresponds to hypotheses involving \strong{coordinated or gregarious movement}, where animals moving in parallel
#'   or in a group (e.g., hunting packs or social gatherings) would produce \strong{fewer intersections} than expected under random conditions.
#' - \code{"Higher"}: This option tests whether the observed intersections are significantly higher than those generated by simulations.
#'   It applies to scenarios where \strong{predatory or chasing interactions} are likely, such as when one trackmaker follows or crosses
#'   the path of another. This behavior results in \strong{more intersections} than what would occur randomly.
#'
#' The selection of the \code{H1} argument must be consistent with the behavioral hypothesis being tested. For example, use \code{"Lower"}
#' when investigating group movement or cooperative behavior, and \code{"Higher"} when analyzing predatory or competitive interactions.
#' The function will automatically adjust the calculation of *p*-values to reflect the selected \code{H1}. If the argument is left
#' \code{NULL}, an error will be triggered, indicating that users must explicitly specify the hypothesis to be tested.
#'
#' The interpretation of the \strong{combined *p*-value} returned by the function is directly influenced by the choice of \code{H1}, as it determines
#' whether the statistical comparison aims to detect a \strong{reduction} or an \strong{increase} in intersection counts compared to the simulated dataset.
#'
#' In addition to hypothesis testing, the \code{track_intersection()} function offers several options for altering the initial positions
#' of simulated tracks through the \code{origin.permutation} argument. The available options include:
#' - \code{"None"}: Simulated trajectories are not shifted and are compared based on their original starting positions.
#' - \code{"Min.Box"}: Trajectories are randomly placed within the \strong{minimum bounding box} surrounding the original starting points.
#' - \code{"Conv.Hull"}: Trajectories are placed within the \strong{convex hull} that encompasses all original starting points, providing
#'   a more precise representation of the area occupied by the tracks.
#' - \code{"Custom"}: Allows users to define a specific region of interest by providing a matrix of coordinates (\code{custom.coord}) that
#'   specifies the vertices of the desired area. This option is particularly useful when certain spatial features or environmental conditions
#'   are known to constrain movement.
#'
#' The choice of \code{origin.permutation} should reflect the nature of the behavioral hypothesis being tested. For example, using
#' \code{"None"} is most appropriate when testing how intersections compare under scenarios where trackmakers originate from specific
#' locations. In contrast, options like \code{"Min.Box"}, \code{"Conv.Hull"}, or \code{"Custom"} are suitable when evaluating how intersections
#' would differ if the tracks originated from a broader or predefined area.
#'
#' The \code{track_intersection()} function also allows for integration with similarity metrics computed using \code{simil_DTW_metric()} and
#' \code{simil_Frechet_metric()}. This combination of intersection counts and similarity metrics can provide a more comprehensive analysis
#' of how trackmakers interacted, whether their movements were coordinated or independent, and whether their interactions were consistent
#' with the hypothesized behavioral patterns.
#'
#' Overall, the selection of \code{H1} and \code{origin.permutation} should be carefully considered in light of the specific hypotheses
#' being tested. By combining intersection metrics with similarity measures, users can obtain a deeper understanding of the
#' behavioral dynamics underlying the observed trackways.
#'
#' @return A \code{track intersection} R object consisting of a list containing the following elements:
#' \item{Intersection_metric}{A matrix of unique intersection counts between trajectories. Each entry
#' represents the number of unique intersection points between the corresponding pair of trajectories.}
#' \item{Intersection_metric_p_values}{(If \code{test = TRUE}) A matrix of *p*-values associated with
#' the intersection metrics, calculated through permutations of simulated trajectory origins. Each entry
#' reflects the probability of observing an intersection count as extreme as the observed one,
#' given the null hypothesis of no difference.}
#' \item{Intersection_metric_p_values_combined}{(If \code{test = TRUE}) A numeric value representing
#' the combined *p*-value for all intersections, indicating the overall significance of the
#' intersection metrics across all pairs of trajectories.}
#' \item{Intersection_metric_simulations}{(If \code{test = TRUE}) A list containing matrices of
#' intersection counts for each simulation iteration, allowing for further inspection of the
#' distribution of intersections across multiple randomized scenarios.}
#'
#' @section Logo:
#' \if{html}{\figure{Logo.png}{options: width=30\%}}
#'
#' @author Humberto G. Ferrón
#' @author humberto.ferron@uv.es
#' @author Macroevolution and Functional Morphology Research Group (www.macrofun.es)
#' @author Cavanilles Institute of Biodiversity and Evolutionary Biology
#' @author Calle Catedrático José Beltrán Martínez, nº 2
#' @author 46980 Paterna - Valencia - Spain
#' @author Phone: +34 (9635) 44477
#'
#' @examples
#' # Example 1: Intersection metrics in the PaluxyRiver dataset.
#' s1 <- simulate_track(PaluxyRiver, nsim = 5, model = "Directed")
#' int1 <- track_intersection(PaluxyRiver, test = TRUE, H1 = "Lower", sim = s1,
#'   origin.permutation = "None")
#' print(int1)
#'
#' # Example 2: Using "Min.Box" origin permutation in PaluxyRiver dataset.
#' s2 <- simulate_track(PaluxyRiver, nsim = 5, model = "Constrained")
#' int2 <- track_intersection(PaluxyRiver, test = TRUE, H1 = "Lower", sim = s2,
#'   origin.permutation = "Min.Box")
#' print(int2)
#'
#' # Example 3: Using "Conv.Hull" origin permutation in PaluxyRiver dataset.
#' s3 <- simulate_track(PaluxyRiver, nsim = 5, model = "Unconstrained")
#' int3 <- track_intersection(PaluxyRiver, test = TRUE, H1 = "Lower", sim = s3,
#'   origin.permutation = "Conv.Hull")
#' print(int3)
#'
#' # Example 4: Using "Min.Box" origin permutation in MountTom subset.
#' sbMountTom <- subset_track(MountTom, tracks = c(1, 2, 3, 4, 7, 8, 9, 13, 15, 16, 18))
#' s4 <- simulate_track(sbMountTom, nsim = 5)
#' int4 <- track_intersection(sbMountTom, test = TRUE, H1 = "Higher", sim = s4,
#'   origin.permutation = "Min.Box")
#' print(int4)
#'
#' # Example 5: Customized origin permutation in MountTom subset.
#' sbMountTom <- subset_track(MountTom, tracks = c(1, 2, 3, 4, 7, 8, 9, 13, 15, 16, 18))
#' s5 <- simulate_track(sbMountTom, nsim = 5)
#' area_origin <- matrix(c(50, 5, 10, 5, 10, 20, 50, 20), ncol = 2, byrow = TRUE)
#' int5 <- track_intersection(sbMountTom, test = TRUE, H1 = "Higher", sim = s5,
#'   origin.permutation = "Custom", custom.coord = area_origin)
#' print(int5)
#'
#' @importFrom grDevices chull
#' @importFrom shotGroups getMinBBox
#' @importFrom splancs csr
#' @importFrom trajr TrajTranslate
#'
#' @seealso \code{\link{tps_to_track}}, \code{\link{simulate_track}}, \code{\link{simil_DTW_metric}}, \code{\link{simil_Frechet_metric}}
#'
#' @export


track_intersection <- function(data, test = NULL, H1 = NULL, sim = NULL, origin.permutation = NULL, custom.coord = NULL) {

  ## Set default values if arguments are NULL----
  if (is.null(test)) test <- FALSE # Set default if 'test' is NULL
  if (is.null(origin.permutation)) origin.permutation <- "None" # Set default if 'origin.permutation' is NULL

  ## Errors and Warnings----

  # Check if 'data' is a list with at least two elements
  if (!is.list(data) || length(data) < 2) {
    stop("The 'data' argument must be a 'track' R object, which is a list consisting of two elements.")
  }

  # Check if the two elements of 'data' are lists
  if (!is.list(data[[1]]) || !is.list(data[[2]])) {
    stop("The two elements of 'data' must be lists.")
  }

  # Warn if the 'test' argument is not a boolean
  if (!is.logical(test)) {
    stop("'test' argument should be TRUE or FALSE.")
  }

  # Check if 'sim' is provided when test is TRUE
  if (test == TRUE && is.null(sim)) {
    stop("A 'sim' argument must be provided when 'test' is TRUE.")
  }

  # Check if 'H1' argument is provided when 'test = TRUE'
  if (test == TRUE && is.null(H1)) {
    stop("'H1' argument must be specified when 'test' is TRUE. Choose either 'Lower' or 'Higher'.")
  }

  # Check if 'H1' argument is one of the valid options
  if (!is.null(H1) && !H1 %in% c("Lower", "Higher")) {
    stop("Invalid 'H1' argument. Valid options are 'Lower' or 'Higher'.")
  }

  # If 'sim' is provided, ensure it is a list and has the same structure as 'data'
  if (!is.null(sim)) {
    if (!is.list(sim)) {
      stop("The 'sim' argument must be a list.")
    }

    # Check that 'sim' contains the same number of tracks as 'data'
    if (length(sim[[1]]) != length(data[[1]])) {
      stop("The 'sim' list must have the same number of trajectories as 'data'.")
    }
  }

  # Check if 'origin.permutation' is valid
  valid_permutations <- c("None", "Min.Box", "Conv.Hull", "Custom")
  if (!origin.permutation %in% valid_permutations) {
    stop(paste("Invalid 'origin.permutation'. Valid options are:", paste(valid_permutations, collapse = ", ")))
  }

  # If 'origin.permutation' is "Custom", check if 'custom.coord' is provided
  if (origin.permutation == "Custom" && is.null(custom.coord)) {
    stop("If 'origin.permutation' is set to 'Custom', the 'custom.coord' must be provided.")
  }

  # Check if 'custom.coord' is a matrix or a data frame with two columns
  if (!is.null(custom.coord) && !is.matrix(custom.coord) && !is.data.frame(custom.coord)) {
    stop("The 'custom.coord' must be a matrix or a data frame.")
  }

  if (!is.null(custom.coord) && ncol(as.matrix(custom.coord)) != 2) {
    stop("The 'custom.coord' must have exactly two columns.")
  }


  ## Code----
  data <- data[[1]]

  # Function to determine if two line segments intersect
  find_intersection <- function(p1, p2, p3, p4) {
    x1 <- p1[1]
    y1 <- p1[2]
    x2 <- p2[1]
    y2 <- p2[2]
    x3 <- p3[1]
    y3 <- p3[2]
    x4 <- p4[1]
    y4 <- p4[2]

    # Compute determinants
    denom <- (x1 - x2) * (y3 - y4) - (y1 - y2) * (x3 - x4)

    # Check if the lines are parallel
    if (denom == 0) {
      return(c(NA, NA)) # Lines are parallel or coincident
    }

    # Compute intersection point
    intersect_x <- ((x1 * y2 - y1 * x2) * (x3 - x4) - (x1 - x2) * (x3 * y4 - y3 * x4)) / denom
    intersect_y <- ((x1 * y2 - y1 * x2) * (y3 - y4) - (y1 - y2) * (x3 * y4 - y3 * x4)) / denom

    # Check if the intersection point is within both line segments
    if (min(x1, x2) <= intersect_x && intersect_x <= max(x1, x2) &&
      min(y1, y2) <= intersect_y && intersect_y <= max(y1, y2) &&
      min(x3, x4) <= intersect_x && intersect_x <= max(x3, x4) &&
      min(y3, y4) <= intersect_y && intersect_y <= max(y3, y4)) {
      return(c(intersect_x, intersect_y))
    } else {
      return(c(NA, NA))
    }
  }

  # Function to count unique intersections between two trajectories
  intersect <- function(traj1, traj2) {
    intersections <- list()

    for (i in 1:(length(traj1$x) - 1)) {
      for (j in 1:(length(traj2$x) - 1)) {
        int <- find_intersection(
          c(traj1$x[i], traj1$y[i]), c(traj1$x[i + 1], traj1$y[i + 1]),
          c(traj2$x[j], traj2$y[j]), c(traj2$x[j + 1], traj2$y[j + 1])
        )

        if (!is.na(int[1])) {
          intersections[[length(intersections) + 1]] <- int
        }
      }
    }

    # Remove duplicate intersections
    unique_intersections <- unique(do.call(rbind, intersections))

    # Return the number of unique intersections
    if (is.null(unique_intersections)) {
      return(0)
    } else {
      return(nrow(unique_intersections))
    }
  }


  # Calculate actual intersection metrics
  Matrixsim <- data.frame(matrix(nrow = length(data), ncol = length(data)))
  colnames(Matrixsim) <- names(data)
  rownames(Matrixsim) <- names(data)

  Intersect <- Matrixsim
  for (i in 1:length(data)) {
    for (j in 1:length(data)) {
      if (i <= j) next
      Intersect[j, i] <- intersect(data[[i]], data[[j]])
    }
  }


  # Calculate simulation intersection metrics
  # Permutation of coordinates at origin
  if (test == TRUE) {
    nsim <- length(sim)

    if (origin.permutation != "None") {
      mat <- matrix(ncol = 2, nrow = length(data))

      for (i in 1:length(data)) {
        mat[, 1][i] <- data[[i]]$x[1]
        mat[, 2][i] <- data[[i]]$y[1]
      }

      if (length(data) == 2) {
        mat <- rbind(mat, mat * 1.000001)
      }

      if (origin.permutation == "Min.Box") {
        mat <- getMinBBox(mat)
        mat <- mat$pts
      }

      if (origin.permutation == "Conv.Hull") {
        mat <- mat[chull(mat), ]
      }

      if (origin.permutation == "Custom") {
        mat <- custom.coord
      }

      for (i in 1:length(sim)) {
        for (j in 1:length(sim[[1]])) {
          sim[[i]][[j]] <- TrajTranslate(sim[[i]][[j]], csr(mat, 1)[1], csr(mat, 1)[2])
        }

        message(paste(Sys.time(), paste("Permutation", i)))
        message(" ")
        message(paste("Permutation of coordinates at origin using", origin.permutation))
        message("------------------------------------")
        if (i == nsim) {
          message("PERMUTATION COMPLETED")
          message("------------------------------------")
          message(" ")
        }
      }
    }

    # Calculate metrics
    Intersectsim <- Matrixsim

    listIntersect <- list()
    listdiffIntersect <- c()

    for (i in 1:nsim) {
      for (c in 1:length(data)) {
        for (r in 1:length(data)) {
          if (c <= r) next
          Intersectsim[r, c] <- intersect(sim[[i]][[r]], sim[[i]][[c]])
        }
      }
      listIntersect[[i]] <- Intersectsim

      diffinter <- c(as.matrix(Intersect - listIntersect[[i]]))
      diffinter <- diffinter[!is.na(diffinter)]
      listdiffIntersect[i] <- sum(diffinter)

      message(paste(Sys.time(), paste("Iteration", i)))
      message(" ")
      message("Intersect metric")
      Intersectsim
      message("------------------------------------")
      if (i == nsim) {
        message("ANALYSIS COMPLETED")
        message("------------------------------------")
        message(" ")
      }
    }

    # Calculate p-values
    if (H1 == "Lower") {
    Intersectsim_pval <- Matrixsim

    vector <- c()
    for (c in 1:length(data)) {
      for (r in 1:length(data)) {
        if (c <= r) next
        for (i in 1:nsim) {
          vector[i] <- listIntersect[[i]][r, c]
          Intersectsim_pval[r, c] <- 1-length(which(vector < Intersect[r, c])) / nsim
        }
      }
    }

    vector <- c()

    Intersect_together_pval <- 1-length(which(listdiffIntersect > 0)) / nsim
    }

    if (H1 == "Higher") {
      Intersectsim_pval <- Matrixsim

      vector <- c()
      for (c in 1:length(data)) {
        for (r in 1:length(data)) {
          if (c <= r) next
          for (i in 1:nsim) {
            vector[i] <- listIntersect[[i]][r, c]
            Intersectsim_pval[r, c] <- 1-length(which(vector > Intersect[r, c])) / nsim
          }
        }
      }

      vector <- c()

      Intersect_together_pval <- 1-length(which(listdiffIntersect < 0)) / nsim
    }
  }


  if (test == TRUE) {
    list <- list()
    list[[1]] <- Intersect
    list[[2]] <- Intersectsim_pval
    list[[3]] <- Intersect_together_pval
    list[[4]] <- listIntersect

    names(list) <- c("Intersection_metric", "Intersection_metric_p_values", "Intersection_metric_p_values_combined", "Intersection_metric_simulations")
    return(list)
  } else {
    return(Intersect)
  }
}
