#
#  repeated : A Library of Repeated Measurements Models
#  Copyright (C) 1998 J.K. Lindsey
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#  SYNOPSIS
#
#     gar(response, distribution="normal", times, totals=NULL, censor=NULL,
#	delta=NULL, mu, shape=NULL, preg=NULL, pshape=NULL,
#	pdepend=NULL, transform="identity", link="identity",
#	autocorr="exponential", order=1, print.level=0, ndigit=10,
#	gradtol=0.00001, steptol=0.00001, fscale=1, iterlim=100,
#	typsiz=abs(p), stepmax=10*sqrt(p%*%p))
#
#  DESCRIPTION
#
#    A function to fit generalized nonlinear autoregression models with
#  various distributions

gar <- function(response, distribution="normal", times, totals=NULL, censor=NULL,
	delta=NULL, mu, shape=NULL, preg=NULL, pshape=NULL,
	pdepend=NULL, transform="identity", link="identity",
	autocorr="exponential", order=1, print.level=0, ndigit=10,
	gradtol=0.00001, steptol=0.00001, fscale=1, iterlim=100,
	typsiz=abs(p), stepmax=10*sqrt(p%*%p)){
likekal <- function(p){
	eta <- mu(p)
	if(sh){
		shr <- shape(p[nprd1:np])
		theta <- c(p[npr1:nprd],exp(p[np]))}
	else theta <- p[npr1:np]
	z <- .C("gar",
		y=y,
		total=zna$response$n,
		my=as.integer(3*max(y)),
		nobs=as.integer(zna$response$nobs),
		nind=as.integer(nind),
		times=as.double(zna$response$times),
		censor=as.integer(censor),
		cens=as.integer(!is.null(zna$response$censor)),
		eta=eta,
		theta=theta,
		model=as.integer(mdl),
		thp=as.integer(thp),
		shape=shr,
		sh=as.integer(sh),
		link=as.integer(lnk),
		ar=as.integer(ar),
		order=as.integer(order),
		pred=double(n),
		rpred=double(n),
		like=double(1))
	z$like+jacob}
call <- sys.call()
tmp <- c("binomial","Poisson","exponential","negative binomial",
	"mult Poisson","double Poisson","beta binomial","mult binomial",
	"double binomial","normal","logistic","Cauchy", "Weibull","gamma",
	"Laplace","inverse Gauss","Pareto","Levy","gen gamma",
	"gen logistic","Hjorth","Burr","gen Weibull","gen extreme value",
	"gen inverse Gauss","power exponential")
mdl <- match(distribution <- match.arg(distribution,tmp),tmp)
tmp <- c("exponential","gaussian","cauchy","spherical","IOU")
ar <- match(autocorr <- match.arg(autocorr,tmp),tmp)
transform <- match.arg(transform,c("identity","exp","square","sqrt","log"))
tmp <- c("identity","exp","square","sqrt","log","logit","cloglog")
lnk <- match(link <- match.arg(link,tmp),tmp)
if((link=="logit"||link=="cloglog")&&(mdl!=1&&mdl!=7&&mdl!=8&&mdl!=9))stop("logit and cloglog links can only be used with binary data")
formula <- NULL
if(is.language(mu)){
	formula <- mu
	mt <- terms(mu)
	if(is.numeric(mt[[2]])){
		dm1 <- matrix(1)
		colnames(dm1) <- "(Intercept)"
		npt1 <- 1
		mu <- function(p) p[1]*rep(1,n)
		nlp <- 1}
	else {
		mf <- model.frame(mt,sys.frame(sys.parent()),na.action=na.fail)
		dm1 <- model.matrix(mt, mf)
		npt1 <- ncol(dm1)
		mu <- function(p) dm1%*%p[1:npt1]
		nlp <- npt1}
	cname <- colnames(dm1)}
else if(missing(mu)){
	mu <- function(p) p[1]*rep(1,n)
	nlp <- 1
	cname <- "(Intercept)"}
else {
	nlp <- NULL
	cname <- NULL
	for(i in 1:length(preg))cname <- c(cname,paste("p",i,sep=""))}
if(missing(times)&&is.matrix(response))times <- matrix(rep(1:ncol(response),nrow(response)),ncol=ncol(response),byrow=T)
if(inherits(response,"response"))zr <- response
else {
	if(mdl==1||mdl==7||mdl==8||mdl==9)zr <- restovec(response, times=times, censor=censor, delta=delta, totals=totals)
	else zr <- restovec(response, times=times, censor=censor, delta=delta)}
if(mdl==1||mdl==7||mdl==8||mdl==9){
	if(is.null(zr$n)){
		if(any(zr$y!=0&&zr$y!=1))stop("responses must be binary if totals are not supplied")
		else zr$n <- rep(1,length(zr$y))}}
zna <- rmna(response=zr)
rm(zr)
y <- zna$response$y
n <- length(y)
if(mdl<10){
	if(any(y<0))stop("all responses must be non-negative")}
else if((mdl!=10)&&(mdl!=11)&&(mdl!=12)&&(mdl!=15)&&(mdl!=18)&&(mdl!=20)&&(mdl!=26)&&(any(y<=0)))
	stop("all responses must be positive")
else if(distribution=="Levy"&&any(zna$response$y<=mu(preg)))
	stop("location function must be strictly less than corresponding observation")
if(distribution=="Pareto"&&pshape<=1)stop("shape parameter must be > 1")
censor <- zna$response$censor
if(is.null(censor))censor <- rep(1,n)
else if(mdl==1||mdl==2||mdl==4||mdl==5||mdl==6||mdl==7||mdl==8||mdl==9)stop(paste("Censored data not allowed for",distribution,"distribution"))
nind <- length(zna$response$nobs)
if(transform=="identity")jacob <- 0
else if(transform=="exp"){
	jacob <- -sum(y[censor==1])
	y <- exp(y)}
else {
	if(any(y<0))stop("Nonpositive response values: invalid transformation")
	else if(transform=="square"){
		jacob <- -sum(log(y[y>0&censor==1]))
		y  <- y^2}
	else if(transform=="sqrt"){
		jacob <- sum(log(y[y>0&censor==1]))/2
		y <- sqrt(y)}
	else if(any(y==0))stop("Zero response values: invalid transformation")
	else if(transform=="log"){
		jacob <- sum(log(y[censor==1]))
		y <- log(y)}}
if(!is.null(zna$response$delta)){
	if(length(zna$response$delta)==1)
		jacob <- jacob-length(y[censor==1])*log(zna$response$delta)
	else jacob <- jacob-sum(log(zna$response$delta[censor==1]))}
npr <- length(preg)
npr1 <- npr+1
if(!is.null(nlp)&&nlp!=npr)stop(paste(nlp,"initial estimates must be supplied in preg"))
if(order!=1&&order!=2)stop("Autoregression must have order 1 or 2")
if(missing(pdepend))
	stop("Initial estimates of the dependence parameters must be supplied")
if(order==2&&length(pdepend)!=2)stop("2 estimates of dependence parameters must be given")
else if(length(pdepend)!=1&&length(pdepend)!=2)
     stop("One or two estimates of dependence parameters must be given")
thp <- length(pdepend)==2&&order==1
nprd <- npr+length(pdepend)
nprd1 <- nprd+1
nps <- length(pshape)
sh <- is.function(shape)
if(length(mu(preg))!=length(zna$response$y))
	stop("The mu function must provide an estimate for each observation")
else if(any(is.na(mu(preg))))
	stop("Non-numerical mu: probably invalid initial values")
if(any(pdepend<=0))stop("All dependence parameters must be positive")
if(!sh){
	if((mdl<=3&&nps!=0)||(mdl>3&&mdl<19&&nps!=1)||(mdl>=19&&nps!=2))
		stop("Incorrect number of shape parameter estimates")
	else if(nps>0&&any(pshape<=0))
		stop("All shape parameters must be positive")
	shr <- rep(0,length(zna$response$y))}
else if(any(is.na(shape(pshape))))
	stop("Non-numerical shape: probably invalid initial values")
else if(length(shape(pshape))!=length(zna$response$y))
	stop("The shape function must provide an estimate for each observation")
p <- c(preg,-log(pdepend))
if(mdl>3){
	if(!sh)p <- c(p,log(pshape))
	else {
	     if(mdl>=19)p <- c(p,pshape[1:(nps-1)],log(pshape[nps]))
	     else p <- c(p,pshape)}}
np <- length(p)
z0 <- nlm(likekal, p, hessian=T, print.level=print.level,
	typsiz=typsiz, ndigit=ndigit, gradtol=gradtol, stepmax=stepmax,
	steptol=steptol, iterlim=iterlim, fscale=fscale)
like <- z0$minimum
if(any(is.na(z0$hessian)))a <- 0
else a <- qr(z0$hessian)$rank
if(a==np)cov <- solve(z0$hessian)
else cov <- matrix(NA,ncol=np,nrow=np)
se <- sqrt(diag(cov))
corr <- cov/(se%o%se)
dimnames(corr) <- list(1:np,1:np)
eta <- mu(z0$estimate)
if(sh){
	shr <- shape(z0$estimate[nprd1:np])
	theta <- c(z0$estimate[npr1:nprd],exp(z0$estimate[np]))}
else theta <- z0$estimate[npr1:np]
z <- .C("gar",
	y=y,
	total=zna$response$n,
	my=as.integer(3*max(y)),
	nobs=as.integer(zna$response$nobs),
	nind=as.integer(nind),
	times=as.double(zna$response$times),
	censor=as.integer(censor),
	cens=as.integer(!is.null(zna$response$censor)),
	eta=eta,
	theta=theta,
	model=as.integer(mdl),
	thp=as.integer(thp),
	shape=shr,
	sh=as.integer(sh),
	link=as.integer(lnk),
	ar=as.integer(ar),
	order=as.integer(order),
	pred=double(n),
	rpred=double(n),
	like=double(1))
if(transform=="exp"){
	z$pred <- log(z$pred)
	z$rpred <- log(z$rpred)}
else if(transform=="square"){
	z$pred  <- sqrt(z$pred)
	z$rpred  <- sqrt(z$rpred)}
else if(transform=="sqrt"){
	z$pred <- z$pred^2
	z$rpred <- z$rpred^2}
else if(transform=="log"){
	z$pred <- exp(z$pred)
	z$rpred <- exp(z$rpred)}
z <- list(
	call=call,
	distribution=distribution,
	mu=mu,
	formula=formula,
	shape=shape,
	response=zna$response,
	link=link,
	order=order,
	autocorr=autocorr,
	transform=transform,
	maxlike=like,
	aic=like+np,
	df=length(zna$response$y)-np,
	np=np,
	npr=npr,
	nps=nps,
	thp=thp,
	coefficients=z0$estimate,
	cname=cname,
	se=se,
	cov=cov,
	corr=corr,
	pred=z$pred,
	rpred=z$rpred,
	grad=z0$gradient,
	iterations=z0$iterations,
	code=z0$code)
class(z) <- c("gar","recursive")
return(z)}

coefficients.gar <- function(z) z$coefficients
deviance.gar <- function(z) 2*z$maxlike
fitted.gar <- function(z, recursive=TRUE)
	if(recursive) z$rpred else z$pred
residuals.gar <- function(z, recursive=TRUE)
	if(recursive) z$response$y-z$rpred else z$response$y-z$pred

print.gar <- function(z, digits = max(3, .Options$digits - 3)) {
	np1 <- if(z$distribution=="binomial"||z$distribution=="exponential"
			||z$distribution=="Poisson") 0
		else if(z$distribution=="gen gamma"
			||z$distribution=="gen logistic"
			||z$distribution=="Hjorth"||z$distribution=="Burr"
			||z$distribution=="gen Weibull"
			||z$distribution=="gen extreme value"
			||z$distribution=="gen inverse Gauss"
			||z$distribution=="power exponential") 2
		else 1
	cat("\nCall:\n",deparse(z$call),"\n\n",sep="")
	if(z$code>2)cat("Warning: no convergence - error",z$code,"\n\n")
	cat("Number of subjects    ",length(z$response$nobs),"\n")
	cat("Number of observations",length(z$response$y),"\n")
	cat("Transformation        ",z$trans,"\n")
	cat("Link function         ",z$link,"\n\n")
	cat(z$distribution,"distribution\n")
	if(z$order==1)cat("First order ")
	else cat("Second order ")
	cat(z$autocorr,"dependence\n")
	cat("\n-Log likelihood   ",z$maxlike,"\n")
	cat("Degrees of freedom",z$df,"\n")
	cat("AIC               ",z$aic,"\n")
	cat("Iterations        ",z$iterations,"\n\n")
	cat("Location parameters\n")
	if(!is.language(z$formula)){
		t <- deparse(z$mu)
		cat(t[2:length(t)],sep="\n")}
	coef.table <- cbind(z$coef[1:z$npr],z$se[1:z$npr])
	dimnames(coef.table) <- list(z$cname, c("estimate","se"))
	print.default(coef.table, digits=digits, print.gap=2)
	if(z$thp||z$order==2){
		cat("\nDependence parameters\n")
		if(z$thp)cname <- c("phi","rho")
		else cname <- c("rho1","rho2")
		coef.table <- cbind(z$coef[(z$npr+1):(z$npr+2)],
			z$se[(z$npr+1):(z$npr+2)],
			exp(-z$coef[(z$npr+1):(z$npr+2)]))}
	else {
		cat("\nDependence parameter\n")
		cname <- "rho"
		coef.table <- cbind(z$coef[z$npr+1],
			z$se[z$npr+1],exp(-z$coef[z$npr+1]))}
	dimnames(coef.table) <- list(cname, c("estimate","se","parameter"))
	print.default(coef.table, digits=digits, print.gap=2)
	if(np1>0){
		cat("\nShape parameters\n")
		if(!is.null(z$shape)){
			t <- deparse(z$shape)
			cat(t[2:length(t)],sep="\n")
			coef.table <- cbind(z$coef[(z$np-z$nps+1):z$np],
				z$se[(z$np-z$nps+1):z$np])
			cname <- NULL
			for(i in 1:nrow(coef.table))
				cname <- c(cname,paste("p",i,sep=""))
			colname <- c("estimate","se")
			if(np1==2){
				cname[length(cname)] <- "psi"
				coef.table <- cbind(coef.table,c(rep(NA,nrow(coef.table)-1),exp(z$coef[z$np])))
				colname <- c(colname,"parameter")}
			dimnames(coef.table) <- list(cname,colname)}
		else {
			cname <- "shape"
			if(np1==2)cname <- c(cname,"psi")
			coef.table <- cbind(z$coef[(z$np-np1+1):z$np],
				z$se[(z$np-np1+1):z$np],
				exp(z$coef[(z$np-np1+1):z$np]))
			dimnames(coef.table) <- list(cname, c("estimate","se","parameter"))}
		print.default(coef.table, digits=digits, print.gap=2)}
	cat("\nCorrelation matrix\n")
	print.default(z$corr, digits=digits)}

plot.residuals.gar <- function(z, x=NULL, subset=NULL, nobs=NULL,
	recursive=T, pch=20, ylab="Residual", xlab=NULL,
	main=NULL, ...){
	if(recursive)z$rresiduals <- z$response$y-z$rpred
	else z$residuals <- z$response$y-z$mu(z$coef)
	plot.residuals.default(z, x=x, subset=subset, nobs=nobs,
		recursive=recursive, pch=pch, ylab=ylab,
		xlab=xlab, main=main, ...)}
