#' Calculate Physiological Efficiency (PE)
#' @description
#' The contribution of fertilizer N from the plant tissues towards the yield component.
#' Formula: PE = (YieldF - Yield0) / (PlantNf - PlantN0)
#' @param YieldF A numeric vector of final yield values.
#' @param Yield0 A numeric vector of non-fertilized control yield values.
#' @param PlantNf A numeric vector of plant N at the end of the experiment.
#' @param PlantN0 A numeric vector of plant N at the beginning/control.
#' @return A numeric vector of PE values.
#' @examples
#' YieldF <- c(12, 13, 14)
#' Yield0 <- c(10, 10, 10)
#' PlantNf <- c(3.5, 4.0, 4.2)
#' PlantN0 <- c(2.0, 2.0, 2.0)
#' PE(YieldF, Yield0, PlantNf, PlantN0)
#' @export
PE <- function(YieldF, Yield0, PlantNf, PlantN0) {
  # Calculate yield difference
  delta_yield <- YieldF - Yield0
  
  # Calculate N difference
  delta_n <- PlantNf - PlantN0
  
  # Avoid division by zero if PlantN doesn't change
  if (any(delta_n == 0)) {
    warning("Some PlantN differences are zero, resulting in Infinite PE values.")
  }
  
  return(delta_yield / delta_n)
}