% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MNM_AR.R
\name{MNM_AR}
\alias{MNM_AR}
\title{Fit a Multi-Species N-Mixture Model with AR-1 Using Nimble}
\usage{
MNM_AR(
  Y = NULL,
  iterations = 60000,
  burnin = 20000,
  thin = 10,
  Xp = NULL,
  Xn = NULL,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{Y}{Array of observed counts, with dimensions (R, T, S, K), where:
\itemize{
\item R: Number of sites.
\item T: Number of repeated counts (replicates).
\item S: Number of species.
\item K: Number of time points.
}}

\item{iterations}{Integer. Number of MCMC iterations to run. Defaults to 60,000.}

\item{burnin}{Integer. Number of iterations to discard as burn-in. Defaults to 20,000.}

\item{thin}{Integer. Thinning interval for saving MCMC samples. Defaults to 10.}

\item{Xp}{Array of detection covariates with dimensions \verb{(R, S, P1)}, where:
\itemize{
\item \code{R}: Number of sites.
\item \code{S}: Number of species.
\item \code{P1}: Number of detection probability covariates.
}}

\item{Xn}{Array of abundance covariates with dimensions \verb{(R, S, K, P2)}, where:
\itemize{
\item \code{R}: Number of sites.
\item \code{S}: Number of species.
\item \code{K}: Number of time points.
\item \code{P2}: Number of abundance covariates.
}}

\item{verbose}{Control the level of output displayed during function execution. Default is TRUE.}

\item{...}{Additional arguments passed for prior distribution specification. Supported distributions include dnorm, dexp, dgamma, dbeta, dunif, dlnorm, dbern, dpois, dbinom, dcat, dmnorm, dwish, dchisq, dinvgamma, dt, dweib, ddirch, dmulti, dmvt. Default prior distributions are:
\itemize{
\item prior_detection_probability: prior distribution for the detection probability intercept (\code{gamma}). Default is \code{'dnorm(0, 0.001)'}.
\item prior_precision: prior distribution for the precision matrix for the species-level random effect. Default is \code{'dwish(Omega[1:S,1:S], df)'}.
\item prior_mean: prior distribution for the mean of the species-level random effect (\code{mu}). Default is \code{'dnorm(0,0.001)'}.
\item prior_hurdle: prior distribution for \code{theta}, the probability of structural zero in hurdle models. Default is \code{'dbeta(1,1)'}.
\item prior_mean_AR: prior distribution for the mean of the autoregressive random effect (\code{phi}). Default is \code{'dnorm(0,0.001)'}.
\item prior_sd_AR: prior distribution for the standard deviation of the autoregressive random effect (\code{phi}). Default is \code{'dexp(1)'}.
}
See Nimble (r-nimble.org) documentation for distribution details.}
}
\value{
An MNM object that contains the following components:
\itemize{
\item summary: Nimble model summary (mean, standard deviation, standard error, quantiles, effective sample size and Rhat value for all monitored values).
\item n_parameters: Number of parameters in the model (for use in calculating information criteria).
\item data: Observed abundances.
\item fitted_Y:Predicted values for Y: posterior predictive checks can be performed by comparing fitted_Y with the observed data.
\item logLik:Log-likelihood of the observed data (Y) given the model parameters.
\item n_converged: Number of parameters with successful convergence (Rhat < 1.1).
\item plot: traceplots and density plots for all monitored variables.
}
}
\description{
Fits a multi-species N-mixture (MNM) model with an autoregressive component (AR-1) using Nimble.
This model is suitable for time-series data collected over long periods and includes covariates for abundance and detection probability.
}
\details{
The model incorporates temporal and species-level covariates and accounts for serial correlation in abundance using an AR(1) process.
The input data should be structured as an array with dimensions (R,T,S,K). See parameter descriptions for details.

Features include:
\itemize{
\item Posterior predictive checks via predicted vs observed abundance.
\item Calculation of log-likelihood for model evaluation.
\item Automatic monitoring of parameter convergence.
}
}
\note{
Ensure that the dimensions of \code{Y}, \code{Xp}, and \code{Xn} match the requirements specified above. Mismatched dimensions will result in errors during model fitting.
}
\examples{
# Example 1:
Y <- array(rpois(1000, lambda = 10), dim = c(10, 10, 5, 2))
Xp <- array(runif(500), dim = c(10, 5, 2, 3))
Xn <- array(runif(1000), dim = c(10, 5, 2, 4))

# Fit the AR-1 model
\donttest{result <- MNM_AR(Y = Y, Xp = Xp, Xn = Xn)}
# nimble creates auxiliary functions that may be removed after model run is complete
# using rm(list = ls(pattern = "^str"))
# Check fitted vs observed abundance
\donttest{plot(result@data, result@fitted_Y)}

data(birds)

# Example 2: North American Breeding Bird Data
# Data must first be reformatted to an array of dimension (R,T,S,K)
R <- 15
T <- 10
S <- 10
K <- 4
# Ensure data is ordered consistently
birds <- birds[order(birds$Route, birds$Year, birds$English_Common_Name), ]

# Create a 4D array with proper dimension
Y <- array(NA, dim = c(R, T, S, K))

# Map route, species, and year to indices
route_idx <- as.numeric(factor(birds$Route))
species_idx <- as.numeric(factor(birds$English_Common_Name))
year_idx <- as.numeric(factor(birds$Year))

# Populate the array
stop_data <- as.matrix(birds[, grep("^Stop", colnames(birds))])

for (i in seq_len(nrow(birds))) {
  Y[route_idx[i], , species_idx[i], year_idx[i]] <- stop_data[i, ]
  }

  # Assign dimnames
  dimnames(Y) <- list(
    Route = sort(unique(birds$Route)),
      Stop = paste0("Stop", 1:T),
        Species = sort(unique(birds$English_Common_Name)),
          Year = sort(unique(birds$Year))
          )

# Selecting only 5 bird species  for analysis:
Y<-Y[,,1:5,]

\donttest{model<-MNM_fit(Y=Y, AR=TRUE, Hurdle=FALSE, iterations=10000, burnin=2000)}

}
\references{
\itemize{
\item Royle, J. A. (2004). N-mixture models for estimating population size from spatially replicated counts. Biometrics, 60(1), 108-115.
\item Mimnagh, N., Parnell, A., Prado, E., & Moral, R. D. A. (2022). Bayesian multi-species N-mixture models for unmarked animal communities. Environmental and Ecological Statistics, 29(4), 755-778.
}
}
\seealso{
\itemize{
\item \code{simulateData}: For generating example datasets compatible with this function.
\item \code{MNM}: For details on creating covariate arrays Xp and Xn.
}
}
