% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MultOrdRS.R
\name{ctrl.multordRS}
\alias{ctrl.multordRS}
\title{Control Function for multordRS}
\usage{
ctrl.multordRS(
  RS = TRUE,
  thresholds.acat = c("full", "shift", "minimal"),
  XforRS = TRUE,
  opt.method = c("L-BFGS-B", "nlminb"),
  Q = 10,
  cores = 5,
  lambda = 0.01
)
}
\arguments{
\item{RS}{Logical value indicating whether response style should be modelled.}

\item{thresholds.acat}{Type of parametrization used for thresholds: \code{thresholds = "full"} implies
separate estimates of threshold values for each response variable; \code{thresholds = "shift"} implies
equal threshold parameter across all response variables modified by shift parameters for
each response variable; \code{thresholds = "minimal"} implies equal threshold parameter across all response variables. This option only applies
for adjacent categories models (\code{model = "acat"} and is not implemented for cumulative models.)}

\item{XforRS}{Logical value indicating whether also covariate effects on the
response style should be considered. Only relevant if \code{RS = TRUE}.}

\item{opt.method}{Specifies optimization algorithm used by \code{\link{optim}}, either
\code{L-BFGS-B} or \code{nlminb}.}

\item{Q}{Number of nodes to be used (per dimension) in Gauss-Hermite-Quadrature. If \code{RS = TRUE},
Gauss-Hermite-Quadrature is two-dimensional.}

\item{cores}{Number of cores to be used in parallelized computation.}

\item{lambda}{Tuning parameter for internal ridge penalty. It is supposed to be set to a small value
to stabilize estimates.}
}
\value{
Returns list of control parameters used in \code{\link{multordRS}}.
}
\description{
Control function for multordRS, a model for multivariate ordinal responses including response styles
}
\examples{
\donttest{
data(tenseness)

## create a small subset of the data to speed up calculations
set.seed(1860)
tenseness <- tenseness[sample(1:nrow(tenseness), 300),]

## scale all metric variables to get comparable parameter estimates
tenseness$Age <- scale(tenseness$Age)
tenseness$Income <- scale(tenseness$Income)

## two formulas, one without and one with explanatory variables (gender and age)
f.tense0 <- as.formula(paste("cbind(",paste(names(tenseness)[1:4],collapse=","),") ~ 1"))
f.tense1 <- as.formula(paste("cbind(",paste(names(tenseness)[1:4],collapse=","),") ~ Gender + Age"))



####
## Adjacent Categories Models
####

## Multivariate adjacent categories model, without response style, 
## without explanatory variables
m.tense0 <- multordRS(f.tense0, data = tenseness, control = ctrl.multordRS(RS = FALSE))
m.tense0

## Multivariate adjacent categories model, with response style as a random effect, 
## without explanatory variables
m.tense1 <- multordRS(f.tense0, data = tenseness)
m.tense1

## Multivariate adjacent categories model, with response style as a random effect, 
## without explanatory variables for response style BUT for location
m.tense2 <- multordRS(f.tense1, data = tenseness, control = ctrl.multordRS(XforRS = FALSE))
m.tense2

## Multivariate adjacent categories model, with response style as a random effect, with 
## explanatory variables for location AND response style
m.tense3 <- multordRS(f.tense1, data = tenseness)
m.tense3

plot(m.tense3)



####
## Cumulative Models
####

## Multivariate cumulative model, without response style, without explanatory variables
m.tense0.cumul <- multordRS(f.tense0, data = tenseness, 
  control = ctrl.multordRS(RS = FALSE), model = "cumulative")
m.tense0.cumul

## Multivariate cumulative model, with response style as a random effect, 
## without explanatory variables
m.tense1.cumul <- multordRS(f.tense0, data = tenseness, model = "cumulative")
m.tense1.cumul

## Multivariate cumulative model, with response style as a random effect, 
## without explanatory variables for response style BUT for location
m.tense2.cumul <- multordRS(f.tense1, data = tenseness, 
  control = ctrl.multordRS(XforRS = FALSE), model = "cumulative")
m.tense2.cumul

## Multivariate cumulative model, with response style as a random effect, with 
## explanatory variables for location AND response style
m.tense3.cumul <- multordRS(f.tense1, data = tenseness, model = "cumulative")
m.tense3.cumul

plot(m.tense3.cumul)

################################################################
## Examples from Schauberger and Tutz (2020) on 
## Data from the German Longitudinal Election Study (GLES) 2017
#################################################################

####
## Source: German Longitudinal Election Study 2017 
## Rossteutscher et al. 2017, https://doi.org/10.4232/1.12927
####

## load GLES data
data(GLES17)

## scale data
GLES17[,7:11] <- scale(GLES17[,7:11])

## define formula
f.GLES <- as.formula(cbind(RefugeeCrisis, ClimateChange, Terrorism, 
                       Globalization, Turkey, NuclearEnergy) ~ 
                       Age + Gender + Unemployment + EastWest + Abitur)

## fit adjacent categories model without and with response style parameters
m.GLES0 <- multordRS(f.GLES, data = GLES17, control =  ctrl.multordRS(RS = FALSE, cores = 6))
m.GLES <- multordRS(f.GLES, data = GLES17, control =  ctrl.multordRS(cores = 6))

m.GLES0
m.GLES

plot(m.GLES, main = "Adjacent categories model")


## fit cumulative model without and with response style parameters (takes pretty long!!!)
m.GLES20 <- multordRS(f.GLES, data = GLES17,  model="cumul", 
control = ctrl.multordRS(opt.method = "nlminb", cores = 6, RS = FALSE))

m.GLES2 <- multordRS(f.GLES, data = GLES17,  model="cumul", 
control = ctrl.multordRS(opt.method = "nlminb", cores = 6))

m.GLES20
m.GLES2

plot(m.GLES2, main = "Cumulative model")

}
}
\references{
Schauberger, Gunther and Tutz, Gerhard (2021): Multivariate Ordinal Random Effects Models Including Subject and Group Specific Response Style Effects, 
\emph{Statistical Modelling}, \doi{10.1177/1471082X20978034}
}
\seealso{
\code{\link{multordRS}} \code{\link{MultOrdRS-package}} \code{\link{plot.MultOrdRS}}
}
\author{
Gunther Schauberger\cr \email{gunther.schauberger@tum.de}\cr
\url{https://orcid.org/0000-0002-0392-1580}
}
\keyword{adjacent}
\keyword{categories}
\keyword{cumulative}
\keyword{multivariate}
\keyword{ordinal}
\keyword{response}
\keyword{style}
