# Maximum Likelihood Estimation for alpha-stable distributions and mixtures
# =============================================================================
# SINGLE STABLE DISTRIBUTION MLE
# =============================================================================


#' Fit Alpha-Stable Distribution using MLE (L-BFGS-B)
#'
#' Estimates the parameters of a single alpha-stable distribution using maximum likelihood
#' and the L-BFGS-B optimization method.
#'
#' @param data Numeric vector of observations.
#' @return Numeric vector of estimated parameters: alpha, beta, gamma, delta.
#' @importFrom stats optim
#' @export
fit_alpha_stable_mle <- function(data) {
  init_params <- stable_fit_init(data)
  start_vals <- c(init_params$alpha, init_params$beta, init_params$gamma, init_params$delta)

  lower_bounds <- c(0.1, -1, 1e-6, -Inf)
  upper_bounds <- c(2.0, 1, Inf, Inf)

  result <- tryCatch({
    optim(par = start_vals,
          fn = negative_log_likelihood,
          data = data,
          method = "L-BFGS-B",
          lower = lower_bounds,
          upper = upper_bounds,
          control = list(maxit = 300))
  }, error = function(e) {
    list(convergence = 1, par = rep(NaN, 4))
  })

  if (result$convergence == 0) {
    return(result$par)
  } else {
    return(rep(NaN, 4))
  }
}

# =============================================================================
# MLE FOR MIXTURE OF TWO STABLE DISTRIBUTIONS
# =============================================================================

#' Fit MLE Mixture of Two Stable Distributions
#'
#' Estimates parameters of a two-component alpha-stable mixture using maximum likelihood
#' and the L-BFGS-B optimization method.
#'
#' @param data Numeric vector of observations.
#' @return Numeric vector of estimated parameters: weight, alpha1, beta1, gamma1, delta1, alpha2, beta2, gamma2, delta2.
#' @importFrom stats optim
#' @export
fit_mle_mixture <- function(data) {
  init_params <- c(0.5, 1.3, 0.0, 1.0, -1.5, 1.7, 0.0, 1.5, 4.5)

  lower_bounds <- c(0.01, 0.1, -1, 1e-2, -Inf, 0.1, -1, 1e-2, -Inf)
  upper_bounds <- c(0.99, 2.0, 1, Inf, Inf, 2.0, 1, Inf, Inf)

  result <- tryCatch({
    optim(par = init_params,
          fn = log_likelihood_mixture,
          data = data,
          method = "L-BFGS-B",
          lower = lower_bounds,
          upper = upper_bounds)
  }, error = function(e) {
    list(convergence = 1, par = init_params)
  })

  if (result$convergence != 0) {
    message("MLE failed: convergence code", result$convergence, "\n")
    return(init_params)
  }

  return(result$par)
}

# =============================================================================
# NEGATIVE LOG-LIKELIHOOD
# =============================================================================


#' Negative log-likelihood for stable distribution using dstable
#'
#' Computes the negative log-likelihood of a stable distribution given parameters and data.
#'
#' @param param Numeric vector of parameters: alpha, beta, gamma, delta.
#' @param obs Numeric vector of observations.
#' @return Scalar value of negative log-likelihood.
#' @importFrom stabledist dstable
#' @export
L_stable <- function(param, obs) {
  tryCatch({
    alpha <- param[1]
    beta  <- param[2]
    gamma <- param[3]
    delta <- param[4]

    pdf_vals <- dstable(obs, alpha = alpha, beta = beta, gamma = gamma, delta = delta, pm = 1)
    pdf_vals <- pmax(pdf_vals, 1e-300)

    return(-sum(log(pdf_vals)))
  }, error = function(e) {
    message("[Error in L_stable]", e$message, "\n")
    return(Inf)
  })
}

# =============================================================================
# MLE VIA NELDER-MEAD
# =============================================================================


#' Maximum likelihood estimation using Nelder-Mead
#'
#' Estimates parameters of a stable distribution using the Nelder-Mead method with penalty constraints.
#'
#' @param x Numeric vector of observations.
#' @return List with estimated alpha, beta, gamma, delta.
#' @importFrom stats optim
#' @export
Max_vrai <- function(x) {
  init_params_list <- stable_fit_init(x)
  init_params <- c(init_params_list$alpha, init_params_list$beta,
                   init_params_list$gamma, init_params_list$delta)

  objective_with_penalty <- function(param, obs) {
    alpha <- param[1]; beta <- param[2]; gamma <- param[3]; delta <- param[4]
    penalty <- 0
    if (alpha < 0.1 || alpha > 2) penalty <- penalty + 1e6
    if (beta < -1 || beta > 1) penalty <- penalty + 1e6
    if (gamma < 1e-3) penalty <- penalty + 1e6
    if (penalty > 0) return(penalty)
    return(L_stable(param, obs))
  }

  result <- tryCatch({
    optim(par = init_params,
          fn = objective_with_penalty,
          obs = x,
          method = "Nelder-Mead",
          control = list(maxit = 10000, reltol = 1e-8))
  }, error = function(e) {
    stop("Optimization failed: ", e$message)
  })

  if (result$convergence != 0) {
    stop("Optimization failed: convergence code ", result$convergence)
  }

  return(list(alpha = result$par[1], beta = result$par[2],
              gamma = result$par[3], delta = result$par[4]))
}

# =============================================================================
# BASIC AND ROBUST MLE WRAPPERS
# =============================================================================


#' Simple MLE estimation with default starting values
#'
#' Estimates stable distribution parameters using Nelder-Mead optimization from default or user-provided starting values.
#'
#' @param X Numeric vector of observations.
#' @param x0 Optional starting values.
#' @return List with estimated alpha, beta, gamma, delta.
#' @importFrom stats optim
#' @export
mle_estimate <- function(X, x0 = NULL) {
  if (is.null(x0)) {
    x0 <- c(1.5, 0.0, 1.0, 0.0)
  }

  result <- tryCatch({
    optim(par = x0,
          fn = negative_log_likelihood,
          data = X,
          method = "Nelder-Mead")
  }, error = function(e) {
    list(convergence = 1, par = rep(NaN, 4))
  })

  if (result$convergence == 0) {
    return(list(alpha = result$par[1], beta = result$par[2],
                gamma = result$par[3], delta = result$par[4]))
  } else {
    return(list(alpha = NaN, beta = NaN, gamma = NaN, delta = NaN))
  }
}


#' Robust MLE estimation with multiple starting points
#'
#' Performs multiple MLE estimations with randomized starting points and selects the best result based on log-likelihood.
#'
#' @param data Numeric vector of observations.
#' @param n_starts Number of random initializations.
#' @return List with best estimated alpha, beta, gamma, delta.
#' @importFrom stats optim runif rnorm sd
#' @export
robust_mle_estimate <- function(data, n_starts = 5) {
  best_result <- list(alpha = NaN, beta = NaN, gamma = NaN, delta = NaN)
  best_loglik <- Inf

  init_result <- tryCatch({
    Max_vrai(data)
  }, error = function(e) NULL)

  if (!is.null(init_result)) {
    loglik <- L_stable(c(init_result$alpha, init_result$beta,
                         init_result$gamma, init_result$delta), data)
    if (is.finite(loglik) && loglik < best_loglik) {
      best_result <- init_result
      best_loglik <- loglik
    }
  }

  for (i in 1:n_starts) {
    x0 <- c(runif(1, 0.5, 2.0),
            runif(1, -0.5, 0.5),
            abs(rnorm(1, sd(data), sd(data)/2)),
            rnorm(1, mean(data), sd(data)))

    result <- tryCatch({
      mle_estimate(data, x0)
    }, error = function(e) NULL)

    if (!is.null(result) && !any(is.na(unlist(result)))) {
      loglik <- L_stable(c(result$alpha, result$beta, result$gamma, result$delta), data)
      if (is.finite(loglik) && loglik < best_loglik) {
        best_result <- result
        best_loglik <- loglik
      }
    }
  }

  return(best_result)
}

# =============================================================================
# USER-FACING WRAPPERS
# =============================================================================


#' Estimate single stable distribution parameters
#'
#' Wrapper function to estimate stable parameters using one of three methods: "basic", "robust", or "lbfgs".
#'
#' @param data Numeric vector of observations.
#' @param method Estimation method: "basic", "robust", or "lbfgs".
#' @return List with estimated alpha, beta, gamma, delta.
#' @export
estimate_stable_params <- function(data, method = "robust") {
  switch(method,
         "basic" = mle_estimate(data),
         "robust" = robust_mle_estimate(data),
         "lbfgs" = {
           params <- fit_alpha_stable_mle(data)
           if (any(is.na(params))) {
             list(alpha = NaN, beta = NaN, gamma = NaN, delta = NaN)
           } else {
             list(alpha = params[1], beta = params[2],
                  gamma = params[3], delta = params[4])
           }
         },
         stop("Unknown method: ", method)
  )
}


#' Estimate mixture of two stable distributions
#'
#' Wrapper function to estimate parameters of a two-component alpha-stable mixture using MLE.
#'
#' @param data Numeric vector of observations.
#' @return List with estimated weight and parameters for both components.
#' @export
estimate_mixture_params <- function(data) {
  params <- fit_mle_mixture(data)

  return(list(
    weight = params[1],
    component1 = list(alpha = params[2], beta = params[3],
                      gamma = params[4], delta = params[5]),
    component2 = list(alpha = params[6], beta = params[7],
                      gamma = params[8], delta = params[9])
  ))
}
