#------------------------------------------------------------------------------


buildVARModel <- function(use, bdata, name, map){
	if(missing(map)){
		latents <- paste0('Latent_', use)
		
		mmat <- emxMeans(x=use, free=FALSE, values=0)
		
		llist <- as.list(use)
		names(llist) <- latents
		lmat <- emxLoadings(x=llist, values=1, free=FALSE)
		
		rmat <- emxResiduals(x=use, free=FALSE, values=0)
		
		ka <- emxMeans(x=latents, free=TRUE, name='LatentMeans')
	} else {
		latents <- names(map)
		
		mmat <- emxMeans(x=use, free=TRUE, values=0, name='Intercepts')
		
		llist <- map
		lmat <- emxLoadings(x=llist)
		# Set first factor loading to 1
		firstL <- sapply(map, '[', 1)
		for(lv in latents){lmat$values[firstL[lv], lv] <- 1; lmat$free[firstL[lv], lv] <- FALSE}
		
		rmat <- emxResiduals(x=use, values=.2, lbound=1e-6)
		
		ka <- emxMeans(x=latents, free=FALSE, values=0, name='LatentMeans')
	}
	
	nlatents <- length(latents)
	ph <- OpenMx::mxMatrix('Diag', nlatents, nlatents, free=TRUE, values=.3, name='Innovations', lbound=1e-6, labels=paste0('Innov', latents))
	
	dy <- OpenMx::mxMatrix('Full', nlatents, nlatents,
		values=diag(.4, nrow=nlatents), free=TRUE,
		labels=paste0('ar_', outer(1:nlatents, 1:nlatents, paste0)),
		lbound=-2.5, ubound=2.5,
		name='Dynamics', dimnames=list(latents, latents))
	
	im <- emxMeans(paste0(latents, '_0'), free=FALSE, name='x0')
	ic <- OpenMx::mxMatrix('Diag', nlatents, nlatents, labels=paste0(latents, '_0'), free=FALSE, values=1, name='P0')
	uu <- OpenMx::mxMatrix('Full', 1, 1, values=1, name='u')
	
	bmodel <- OpenMx::mxModel(name=name,
		lmat, rmat, mmat, ka, ph, dy, im, ic, uu,
		bdata,
		mxExpectationStateSpace(
			A=slot(dy, 'name'),
			B=slot(ka, 'name'),
			C=slot(lmat, 'name'),
			D=slot(mmat, 'name'),
			Q=slot(ph, 'name'),
			R=slot(rmat, 'name'),
			x0=slot(im, 'name'),
			P0=slot(ic, 'name'),
			u=slot(uu, 'name')),
		mxFitFunctionML()
	)
}

emxVARModel <- function(model, data, name, run=FALSE, use, ID) {
	model=c('diag', 'full')
	# TODO lags!
	if(missing(name)) {name <- 'VARModel'}
	data <- data[,use, drop=FALSE]
	bdata <- OpenMx::mxData(data, 'raw')
	
	m <- buildVARModel(use=use, bdata=bdata, name=name)
	if(run){return(mxRun(m))} else {return(m)}
}

emxModelVAR <- emxVARModel

# VAR model might be diagonal, full, structural VAR ... hmm ... maybe a separate function for the "structural VAR"


emxLVARModel <- function(model, data, name, run=FALSE, use, ID) {
	# model=same as factor model
	if(missing(name)){name <- 'LatentVARModel'}
	if(missing(use)){
		use <- unlist(model)
		use <- use[!duplicated(use)]
	}
	# TODO lags!
	data <- data[,use, drop=FALSE]
	bdata <- OpenMx::mxData(data, 'raw')
	
	m <- buildVARModel(use=use, bdata=bdata, name=name, map=model)
	if(run){return(mxRun(m))} else {return(m)}
}

emxModelLVAR <- emxLVARModel


#require(EasyMx)
#data(myFADataRaw)
#ds0 <- myFADataRaw
#lvm <- emxLVARModel(data=ds0, model=list(F1=paste0('x', 1:5), Y=paste0('y', 1:3)), name='lvarmodel')
#lvm <- emxLVARModel(model=list(L1=names(ds0)[1], L2=names(ds0)[2], L3=names(ds0)[3]), data=ds0, use=names(ds0)[c(1, 2, 3)], name='lv')

# Should the model here just be the measurement model and always do the full VAR at the latent level?
#

emxARMAModel <- function(model, data, name, run=FALSE, use=colnames(data)){
    Theta <- NULL
    Q <- NULL
    r <- max(model + c(0, 1))
    I <- diag(1, nrow=r-1)
    O <- matrix(0, nrow=1, ncol=r-1)
    phi <- matrix(c(rep(.3, model[1]), rep(0, r-model[1])), nrow=r, ncol=1)
    theta <- matrix(c(1, rep(.2, model[2]), rep(0, r-model[2]-1)), nrow=r, ncol=1)
    A <- cbind(phi, rbind(I, O))
    afree <- (A != 0) & (A != 1)
    tfree <- (theta !=0) & (theta != 1)
    amat <- mxMatrix('Full', nrow=r, ncol=r, values=A, free=afree, name='A')
    if(model[1] > 0) amat$labels[amat$free] <- paste0('ar', 1:model[1])
    bmat <- mxMatrix('Zero', nrow=r, ncol=1, name='B')
    cmat <- mxMatrix('Full', nrow=1, ncol=r, values=c(1, O), name='C', dimnames=list(use, paste0('x', 1:r)))
    dmat <- mxMatrix('Full', nrow=1, ncol=1, free=TRUE, values=0, name='D', labels='intercept')
    tmat <- mxMatrix('Full', nrow=r, ncol=1, values=theta, free=tfree, name='Theta')
    if(model[2] > 0) tmat$labels[tmat$free] <- paste0('ma', 1:model[2])
    smat <- mxMatrix('Symm', nrow=1, ncol=1, values=.5, free=TRUE, name='Q', labels='sigma2')
    qmat <- mxAlgebra(Theta %&% Q, name='RQR')
    rmat <- mxMatrix('Zero', nrow=1, ncol=1, name='E')
    pmat <- mxMatrix('Diag', nrow=r, ncol=r, values=1e0, name='P0')
    xmat <- mxMatrix('Zero', nrow=r, ncol=1, name='x0')
    umat <- mxMatrix('Unit', nrow=1, ncol=1, name='u')
    model <- mxModel(paste0('ARMA(', model[1], ', ', model[2], ')'),
        amat, bmat, cmat, dmat, tmat, smat, smat, qmat, rmat, pmat, xmat, umat,
        mxData(observed=data, 'raw'),
        mxExpectationStateSpace(A='A', B='B', C='C', D='D', Q='RQR', R='E', x0='x0', P0='P0', u='u'),
        mxFitFunctionML())
    if(run) model <- mxRun(model)
    return(model)
}

emxModelARMA <- emxARMAModel


# emxHarveyModel <- function(model, data, name, run=FALSE, use, ID) {
# 	random walk
# 		drift='zero', measurement='fixed'
# 	random walk with drift
# 		drift='constant', measurement='fixed'
# 	local level = random walk with measurement noise
# 		drift='zero', measurement='stochastic'
# 	stochastic trend = random walk with autoregressive drift
# 		drift='AR', measurement='fixed'
# 	local linear trend = random walk with autoregressive drift and measurement noise
# 		drift='AR', measurement='stochastic'
# 	drift = c('zero', 'constant', 'AR')
# 	measurement = c('fixed', 'stochastic')
	
# }

#emxStateSpaceModel <- function(model, data, name, run=FALSE, use, ID, time, ...) {
#	stop('This function is not yet implemented')
#}


emxModelByID <- function(model, data, name, run=FALSE, ID, equal=c("none", "labels", "all"), ...){
	equal <- match.arg(equal)
	dsList <- listifyData(data, ID)
	if(missing(name)) name <- 'model'
	ids <- names(dsList)
	if(equal == 'all'){
		# Dummy label all free parameters in model
		model <- omxSetParameters(model, labels=names(coef(model)), newlabels=names(coef(model)))
	}
	idnames <- paste0(name, ids)
	moList <- lapply(1L:length(dsList),
		function(i){mxModel(model, name=idnames[i], mxData(dsList[[i]], 'raw'))})
	if(equal == 'none'){
		if(run){moList <- lapply(moList, function(m){mxRun(m)})}
		return(moList)
	} else if(equal %in% c('labels', 'all')){
		m <- mxModel(name=name, moList, mxFitFunctionMultigroup(idnames))
		if(run){m <- mxRun(m)}
		return(m)
	}
	else {stop(paste0("Uknown 'equal' argument '", equal, "'"))}
}

listifyData <- function(data, ID){
	if(is.data.frame(data)){
		# Split data into list by ID
		ids <- unique(data[,ID])
		pdim <- length(ids)
		dsList <- list()
		for(p in 1:pdim){
			dsList[[p]] <- data[data[,ID] == ids[p],]
		}
	} else if(is.list(data)){
		pdim <- length(data)
		ids <- 1:pdim #sapply(sapply(data, '[', ID), '[', i=1)
		dsList <- data
	} else {stop("'data' argument should be a list of data.frame objects or a single data.frame with ID column")}
	names(dsList) <- ids
	return(dsList)
}

emxStateSpaceMixtureModel <- function(model, data, name, run=FALSE, use, ID, time, ...) {
	# model = "VAR", a single MxModel, a list of MxModel objects
	# mix = c('dynamics', 'measurement', 'all')
	
	if(missing(use)){
		# extract dimnames from all 'C' matrices in model list and use the union of those
	}
	
	dsList <- listifyData(data, ID)
	ids <- names(dsList)
	pdim <- length(ids)
	
	# Create a model for each person in each mixture class
	kmods <- if(is.list(model)) model else {stop("'model' argument should be a list of models")}
	kdim <- length(model)
	pkmodels <- list()
	pk <- 1
	for(p in 1:pdim){
	    for(k in 1:kdim){
	        pkmodels[[pk]] <- mxModel(kmods[[k]],
	            name=paste0('Person', ids[p], 'Klass', k),
	            mxData(dsList[[p]], 'raw'),
	            mxFitFunctionML(vector=TRUE))
	        pk <- pk + 1
	    }
	}
	names(pkmodels) <- sapply(pkmodels, slot, 'name')
	
	# Mixture-specific matrix
	if(kdim == 1) {kprob <- 1} else if(kdim == 2) {kprob <- c(2/3, 1/3)} else {kprob <- c(1/(kdim-1), rep((1 - 1/(kdim-1))/(kdim-1), kdim-1))}
	#c(.5, .25, .25)
	kmat <- mxMatrix('Full', nrow=kdim, ncol=1, values=log(kprob/min(kprob)),
	labels=paste0('klass_prob', 1:kdim), free=c(rep(TRUE, kdim-1), FALSE), name='K')
	
	# Create a mixture model for each person that contains the class-person
	#  combined models for that person
	pmodels <- list()
	for(p in 1:pdim){
	    pind <- grep(paste0('Person', ids[p], 'Klass'), names(pkmodels))
	    pmodels[[p]] <- mxModel(model=paste0('Person', ids[p]), pkmodels[pind],
	        kmat,
	        mxExpectationMixture(
	            components=names(pkmodels)[pind],
	            weights='K', scale='softmax'),
	        mxFitFunctionML()
	    )
	}
	names(pmodels) <- sapply(pmodels, slot, 'name')
	
	# Create a multigroup model composed of all the mixture models for each person
	grpmodel <- mxModel(model='container', pmodels,
	    mxFitFunctionMultigroup(names(pmodels))
	)
	
	if(run){return(mxRun(grpmodel))} else {return(grpmodel)}
}


emxModelStateSpaceMixture <- emxStateSpaceMixtureModel


emxStateSpaceMixtureClassify <- function(model){
	pdim <- length(model$submodels)
	pknames <- sapply(sapply(model$submodels, slot, name='submodels'), slot, name='name')
	kdim <- length(pknames)/pdim
	kest <- mxEvalByName(paste0(slot(model$submodels[[1]], name='name'), '.K'), model)
	kestp <- exp(kest)/sum(exp(kest))
	# TODO generalize this for other scalings
	# TODO check that model wasRun
	# TODO check that input model is something like a state space mixture model
	
	# matrix of fit function names for every person-class combination
	pkmat <- matrix(
	paste0(pknames, '.fitfunction'),
	nrow=pdim, ncol=kdim, byrow=TRUE)
	
	# row likelihoods for every person-class-time
	likArray <- apply(pkmat, c(1, 2), mxEvalByName, model=model, simplify=FALSE)
	# dim 1: tdim number of time points
	# dim 2: pdim number of people
	# dim 3: kdim number of classes
	
	# minus 2 log likelihood for each person in every class
	m2llMat <- apply(likArray, c(1, 2), function(x){ -2*sum(log(x[[1]]))})
	# dim 1: pdim number of people
	# dim 2: kdim number of classes
	
	# combine minus 2 log lik with prior prob of each class
	#  in the log scale
	fullM2llMat <- m2llMat + 
	    matrix(-2*log(kestp), nrow=pdim, ncol=kdim, byrow=TRUE)
	
	# best fitting class for each person
	est_klasses <- apply(fullM2llMat, 1, which.min)
	
	# return object
	ret <- list(estimated_classes=est_klasses,
		joint_m2ll=fullM2llMat, m2ll=m2llMat, likelihood=likArray)
	
	return(ret)
}

