\name{predict.DDPstar}
\alias{predict.DDPstar}
\title{
Predictions from fitted DDPstar models
}
\description{
Takes a fitted \code{DDPstar} object produced by \code{DDPstar()} and produces predictions for different functionals given a new set of values for the model covariates. 
}
\usage{
\method{predict}{DDPstar}(object, what = NULL, newdata, reg.select = NULL, den.grid = NULL, 
  quant.probs = NULL, q.value = NULL, 
  parallel = c("no", "multicore", "snow"), ncpus = 1, cl = NULL, ...)
}
\arguments{
\item{object}{An object of class \code{DDPstar} as produced by \code{DDPstar()}.}
\item{what}{Character vector with the functionals to be obtained: ``regfun'' (regression function); ``varfun'' (variance function); ``quantfun'' (condional quantiles); ``denfun'' (conditional densities); ; ``probfun'' (conditional probabilities, i.e., probabilities of the form \eqn{P(Y \leq y^{*} \mid X = x)}).}
\item{newdata}{Data frame containing the values of the covariates at which predictions will be computed}
\item{reg.select}{Numeric value containing the position of the covariate effect in the DDPstar formula for which predictions are required. If NULL (and \code{"regfun"} is in input argument \code{what}), the regression function (coditional mean) is computed.}
\item{den.grid}{Numeric vector with the response variable values at which the conditional densities should be computed. Applicable and required if ``denfun'' is in argument \code{what}. If required and NULL, a sequence of 100 values between the minumum and the maximum of the observed responses is considered.}
\item{quant.probs}{Numeric vector with the quantiles at which to obtain/predict the conditional quantiles. Applicable and required if ``quantfun'' is in argument \code{what}. If required and NULL, the deciles are computed.}
\item{q.value}{Numeric vector with the response variable values at which to obtain/predict the conditional probabilities. Applicable and required if ``probfun'' is in argument \code{what}.}
\item{parallel}{A characters string with the type of parallel operation: either "no" (default), "multicore" (not available on Windows) or "snow".}
\item{ncpus}{An integer with the number of processes to be used in parallel operation. Defaults to 1.}
\item{cl}{An object inheriting from class \code{cluster} (from the \code{parallel} package), specifying an optional parallel or snow cluster if parallel = "snow". If not supplied, a cluster on the local machine is created for the duration of the call.}
\item{\dots}{Further arguments passed to or from other methods. Not yet implemented.}
}
\value{
A list with the following components (\code{npred} is the number of columns in \code{newdata} and \code{nsave} is the number of Gibbs sampler iterations saved)
\item{regfun}{Present if \code{"regfun"} is in input argument \code{what}. A matrix of dimension \code{npred}\eqn{\times}\code{nsave} with the MCMC realisations of the regression function (conditional mean) or the desired component (if \code{reg.select} is used). }
\item{reg.select}{Argument \code{reg.select} used in the call. Present if \code{"regfun"} is in input argument \code{what}.}
\item{varfun}{Present if \code{"varfun"} is in input argument \code{what}. A matrix of dimension \code{npred}\eqn{\times}\code{nsave} with the MCMC realisations of the conditional variance.}
\item{quantfun}{Present if \code{"quantfun"} is in input argument \code{what}. An array of dimension \code{npred}\eqn{\times}\code{nsave}\eqn{\times}\code{nq} with the MCMC realisations of the conditional quantiles. Here \code{nq} is the length of argument \code{quant.probs}.}
\item{quant.probs}{Argument \code{quant.probs} used in the call. Present if \code{"quantfun"} is in input argument \code{what}.}
\item{denfun}{Present if \code{"denfun"} is in input argument \code{what}. An array of dimension \code{nden}\eqn{\times}\code{npred}\eqn{\times}\code{nsave} with the MCMC realisations of the conditional densities. Here \code{nden} is the length of argument \code{den.grid}.}
\item{den.grid}{Argument \code{den.grid} used in the call. Present if \code{"denfun"} is in input argument \code{what}.}
\item{probfun}{Present if \code{"probfun"} is in input argument \code{what}. An array of dimension \code{ny}\eqn{\times}\code{npred}\eqn{\times}\code{nsave} with the MCMC realisations of the conditional probabilities. Here \code{ny} is the length of argument \code{q.value}}
\item{q.value}{Argument \code{q.value} used in the call. Present if \code{"probfun"} is in input argument \code{what}.}
}
\seealso{
\code{\link{DDPstar}}
}
\examples{
library(DDPstar)
data(dde)
dde$GAD <- dde$GAD/7 # GAD in weeks
\donttest{
set.seed(10) # For reproducibility
fit_dde <- DDPstar(formula = GAD ~ f(DDE, bdeg = 3, nseg = 20, pord = 2, atau = 1, btau = 0.005), 
 data = dde, mcmc = list(nburn = 20000, nsave = 15000, nskip = 1), 
  prior = list(a = 2, b = 0.5, aalpha = 2, balpha = 2, L = 20), 
  standardise = TRUE, compute.lpml = TRUE, compute.WAIC = TRUE)
 
# Data frame for predictions (regression, variance, quartiles and probabilities functions)
df_pred <- data.frame(DDE = seq(min(dde$DDE), max(dde$DDE), length = 50))

# Data frame for conditional densities
# Compute the densities for 4 different DDE values
df_pred_den <- data.frame(DDE = quantile(dde$DDE, c(0.1, 0.6, 0.9, 0.99)))
sequenceGAD <- seq(min(dde$GAD), max(dde$GAD), length = 100)
 
# Regression and variance function
  reg_var_DDE <- predict(fit_dde, newdata = df_pred, 
    what = c("regfun", "varfun"), 
    parallel = "multicore", 
    ncpus = 2)
    # Regression function
    reg_m  <- apply(reg_var_DDE$regfun, 1, median) 
    reg_ql <- apply(reg_var_DDE$regfun, 1, quantile, 0.025) 
    reg_qh <- apply(reg_var_DDE$regfun, 1, quantile, 0.975)

    plot(dde$DDE, dde$GAD, xlab = "DDE (mg/L)", 
      ylab = "Gestational age at delivery (in weeks)",
      main = "Regression function")
    lines(df_pred$DDE, reg_m, lwd = 2)
    lines(df_pred$DDE, reg_ql, lty = 2)
    lines(df_pred$DDE, reg_qh, lty = 2)

    # Variance function
    var_m  <- apply(reg_var_DDE$varfun, 1, median) 
    var_ql <- apply(reg_var_DDE$varfun, 1, quantile, 0.025) 
    var_qh <- apply(reg_var_DDE$varfun, 1, quantile, 0.975)

    plot(df_pred$DDE, var_m, type = "l", lwd = 2,
      xlab = "DDE (mg/L)", 
      ylab = "",
      main = "Variance function")
    lines(df_pred$DDE, var_ql, lty = 2)
    lines(df_pred$DDE, var_qh, lty = 2)

# Quartiles
  p <- c(0.25, 0.5, 0.75)
  quart_DDE <- predict(fit_dde, newdata = df_pred, 
    what = "quantfun", quant.probs = p, 
    parallel = "multicore", ncpus = 2)
  
  quart_m <- apply(quart_DDE$quantfun , c(1,3), median)
  quart_ql <- apply(quart_DDE$quantfun , c(1,3), quantile, 0.025)
  quart_qh <- apply(quart_DDE$quantfun , c(1,3), quantile, 0.975)
  
  plot(dde$DDE, dde$GAD, xlab = "DDE (mg/L)", 
      ylab = "Gestational age at delivery (in weeks)",
      main = "Conditional quartiles")
  
  # Q1    
  lines(df_pred$DDE, quart_m[,1], lwd = 2, col = 2)
  lines(df_pred$DDE, quart_ql[,1], lty = 2, col = 2)
  lines(df_pred$DDE, quart_qh[,1], lty = 2, col = 2)
  
  # Median
  lines(df_pred$DDE, quart_m[,2], lwd = 2)
  lines(df_pred$DDE, quart_ql[,2], lty = 2)
  lines(df_pred$DDE, quart_qh[,2], lty = 2)
  
  # Q3
  lines(df_pred$DDE, quart_m[,3], lwd = 2, col = 3)
  lines(df_pred$DDE, quart_ql[,3], lty = 2, col = 3)
  lines(df_pred$DDE, quart_qh[,3], lty = 2, col = 3)

# Conditional densities
  den_DDE <- predict(fit_dde, newdata = df_pred_den,
    what = "denfun", den.grid = sequenceGAD,
    parallel = "multicore", ncpus = 2)
  
  den_m  <- apply(den_DDE$denfun, c(1,2), median)
  den_ql <- apply(den_DDE$denfun, c(1,2), quantile, 0.025)
  den_qh <- apply(den_DDE$denfun, c(1,2), quantile, 0.975)
  
  op <- par(mfrow = c(2,2))
  plot(sequenceGAD, den_m[,1], type = "l", lwd = 2,
    xlab = "DDE (mg/L)",
    ylab = "",
    main = "DDE = 12.57 (10th percentile)")
  lines(sequenceGAD, den_ql[,1], lty = 2)
  lines(sequenceGAD, den_qh[,1], lty = 2)

  plot(sequenceGAD, den_m[,2], type = "l", lwd = 2,
  xlab = "DDE (mg/L)",
    ylab = "",
    main = "DDE = 28.44 (60th percentile)")
  lines(sequenceGAD, den_ql[,2], lty = 2)
  lines(sequenceGAD, den_qh[,2], lty = 2)

  plot(sequenceGAD, den_m[,3], type = "l", lwd = 2,
    xlab = "DDE (mg/L)",
    ylab = "",
    main = "DDE = 53.72 (90th percentile)")
  lines(sequenceGAD, den_ql[,3], lty = 2)
  lines(sequenceGAD, den_qh[,3], lty = 2)
  plot(sequenceGAD, den_m[,4], type = "l", lwd = 2,
    xlab = "DDE (mg/L)",
    ylab = "",
    main = "DDE = 105.47 (99th percentile)")
  lines(sequenceGAD, den_ql[,4], lty = 2)
  lines(sequenceGAD, den_qh[,4], lty = 2) 
  par(op)

# Conditional probabilities
  probs_DDE <- predict(fit_dde, newdata = df_pred, 
    what = "probfun", q.value = c(37), 
    parallel = "multicore", ncpus = 2)
  
  prob_m  <- apply(probs_DDE$probfun, c(1,2), median)
  prob_ql <- apply(probs_DDE$probfun, c(1,2), quantile, 0.025)
  prob_qh <- apply(probs_DDE$probfun, c(1,2), quantile, 0.975)
 
  plot(df_pred$DDE, prob_m, type = "l", lwd = 2, ylim = c(0, 1),
      xlab = "DDE (mg/L)", 
      ylab = "P(GAD < 37 | DDE)",
      main = "Conditional Probabilities (premature delivery)")
  lines(df_pred$DDE, prob_ql, lty = 2)
  lines(df_pred$DDE, prob_qh, lty = 2)
}

\dontshow{
mcmc <- list(nburn = 20, nsave = 15, nskip = 1)
set.seed(10) # For reproducibility
fit_dde <- DDPstar(formula = GAD ~ f(DDE, bdeg = 3, nseg = 20, pord = 2, atau = 1, btau = 0.005), 
 data = dde, mcmc = mcmc, prior = list(a = 2, b = 0.5, aalpha = 2, balpha = 2, L = 20), 
 standardise = TRUE, compute.lpml = TRUE, compute.WAIC = TRUE)

# Data frame for predictions (regression, variance, quartiles and probabilities functions)
df_pred <- data.frame(DDE = seq(min(dde$DDE), max(dde$DDE), length = 50))

# Data frame for conditional densities
# Compute the densities for 4 different DDE values
sequenceGAD <- seq(min(dde$GAD), max(dde$GAD), length = 100)
df_pred_den <- data.frame(DDE = quantile(dde$DDE, c(0.1, 0.6, 0.9, 0.99)))

# Regression and variance function
reg_var_DDE <- predict(fit_dde, newdata = df_pred, 
  what = c("regfun", "varfun"), 
  parallel = "multicore", 
  ncpus = 2)

  # Regression function
  reg_m  <- apply(reg_var_DDE$regfun, 1, median) 
  reg_ql <- apply(reg_var_DDE$regfun, 1, quantile, 0.025) 
  reg_qh <- apply(reg_var_DDE$regfun, 1, quantile, 0.975)

  plot(dde$DDE, dde$GAD, xlab = "DDE (mg/L)", 
    ylab = "Gestational age at delivery (in weeks)",
    main = "Regression function")
  lines(df_pred$DDE, reg_m, lwd = 2)
  lines(df_pred$DDE, reg_ql, lty = 2)
  lines(df_pred$DDE, reg_qh, lty = 2)

  # Variance function
  var_m  <- apply(reg_var_DDE$varfun, 1, median) 
  var_ql <- apply(reg_var_DDE$varfun, 1, quantile, 0.025) 
  var_qh <- apply(reg_var_DDE$varfun, 1, quantile, 0.975)

  plot(df_pred$DDE, var_m, type = "l", lwd = 2,
    xlab = "DDE (mg/L)", 
    ylab = "",
    main = "Variance function")
  lines(df_pred$DDE, var_ql, lty = 2)
  lines(df_pred$DDE, var_qh, lty = 2)

# Quartiles
p <- c(0.25, 0.5, 0.75)
quart_DDE <- predict(fit_dde, newdata = df_pred, 
  what = "quantfun", quant.probs = p, 
  parallel = "multicore", ncpus = 2)
quart_m <- apply(quart_DDE$quantfun , c(1,3), median)
quart_ql <- apply(quart_DDE$quantfun , c(1,3), quantile, 0.025)
quart_qh <- apply(quart_DDE$quantfun , c(1,3), quantile, 0.975)

plot(dde$DDE, dde$GAD, xlab = "DDE (mg/L)", 
    ylab = "Gestational age at delivery (in weeks)",
    main = "Conditional quartiles")
# Q1    
lines(df_pred$DDE, quart_m[,1], lwd = 2, col = 2)
lines(df_pred$DDE, quart_ql[,1], lty = 2, col = 2)
lines(df_pred$DDE, quart_qh[,1], lty = 2, col = 2)
# Median
lines(df_pred$DDE, quart_m[,2], lwd = 2)
lines(df_pred$DDE, quart_ql[,2], lty = 2)
lines(df_pred$DDE, quart_qh[,2], lty = 2)
# Q3
lines(df_pred$DDE, quart_m[,3], lwd = 2, col = 3)
lines(df_pred$DDE, quart_ql[,3], lty = 2, col = 3)
lines(df_pred$DDE, quart_qh[,3], lty = 2, col = 3)

# Conditional densities
den_DDE <- predict(fit_dde, newdata = df_pred_den,
  what = "denfun", den.grid = sequenceGAD,
  parallel = "multicore", ncpus = 2)

den_m  <- apply(den_DDE$denfun, c(1,2), median)
den_ql <- apply(den_DDE$denfun, c(1,2), quantile, 0.025)
den_qh <- apply(den_DDE$denfun, c(1,2), quantile, 0.975)

op <- par(mfrow = c(2,2))
plot(sequenceGAD, den_m[,1], type = "l", lwd = 2,
  xlab = "DDE (mg/L)",
  ylab = "",
  main = "DDE = 12.57 (10th percentile)")
lines(sequenceGAD, den_ql[,1], lty = 2)
lines(sequenceGAD, den_qh[,1], lty = 2)

plot(sequenceGAD, den_m[,2], type = "l", lwd = 2,
  xlab = "DDE (mg/L)",
  ylab = "",
  main = "DDE = 28.44 (60th percentile)")
lines(sequenceGAD, den_ql[,2], lty = 2)
lines(sequenceGAD, den_qh[,2], lty = 2)

plot(sequenceGAD, den_m[,3], type = "l", lwd = 2,
  xlab = "DDE (mg/L)",
  ylab = "",
  main = "DDE = 53.72 (90th percentile)")
lines(sequenceGAD, den_ql[,3], lty = 2)
lines(sequenceGAD, den_qh[,3], lty = 2)

plot(sequenceGAD, den_m[,4], type = "l", lwd = 2,
  xlab = "DDE (mg/L)",
  ylab = "",
  main = "DDE = 105.47 (99th percentile)")
lines(sequenceGAD, den_ql[,4], lty = 2)
lines(sequenceGAD, den_qh[,4], lty = 2) 
par(op)

# Conditional probabilities
probs_DDE <- predict(fit_dde, newdata = df_pred, 
  what = "probfun", q.value = c(37), 
  parallel = "multicore", ncpus = 2)
prob_m  <- apply(probs_DDE$probfun, c(1,2), median)
prob_ql <- apply(probs_DDE$probfun, c(1,2), quantile, 0.025)
prob_qh <- apply(probs_DDE$probfun, c(1,2), quantile, 0.975)

plot(df_pred$DDE, prob_m, type = "l", lwd = 2, ylim = c(0, 1),
    xlab = "DDE (mg/L)", 
    ylab = "P(GAD < 37 | DDE)",
    main = "Conditional Probabilities (premature delivery)")
lines(df_pred$DDE, prob_ql, lty = 2)
lines(df_pred$DDE, prob_qh, lty = 2)
}
}