# ChinAPIs - Access Chinese Data via APIs and Curated Datasets
# Version 0.1.1
# Copyright (c) 2026 Renzo Caceres Rossi
# Licensed under the MIT License.
# See the LICENSE file in the root directory for full license text.


#' Get Hospital Beds per 1,000 People in China from World Bank
#'
#' @description
#' Retrieves data on the number of hospital beds per 1,000 people in China
#' from 2010 to 2022 using the World Bank Open Data API.
#' The indicator used is \code{SH.MED.BEDS.ZS}.
#'
#' @return A tibble with the following columns:
#' \itemize{
#'   \item \code{indicator}: Indicator name (e.g., "Hospital beds (per 1,000 people)")
#'   \item \code{country}: Country name ("China")
#'   \item \code{year}: Year of the data (integer)
#'   \item \code{value}: Hospital beds per 1,000 people (numeric)
#' }
#'
#' @details
#' This function sends a GET request to the World Bank API.
#' If the API request fails or returns an error status code,
#' the function returns \code{NULL} with an informative message.
#'
#' @note Requires internet connection.
#'
#' @source World Bank Open Data API: \url{https://data.worldbank.org/indicator/SH.MED.BEDS.ZS}
#'
#' @examples
#' if (interactive()) {
#'   get_china_hospital_beds()
#' }
#'
#' @seealso \code{\link[httr]{GET}}, \code{\link[jsonlite]{fromJSON}}, \code{\link[dplyr]{as_tibble}}
#'
#' @importFrom httr GET content
#' @importFrom jsonlite fromJSON
#' @importFrom dplyr as_tibble
#'
#' @export
get_china_hospital_beds <- function() {

  url <- "https://api.worldbank.org/v2/country/CHN/indicator/SH.MED.BEDS.ZS?format=json&date=2010:2022&per_page=100"

  result <- tryCatch({

    res <- httr::GET(url, httr::timeout(10))

    if (httr::status_code(res) != 200) {
      message("World Bank API returned status: ", httr::status_code(res))
      return(NULL)
    }

    content <- jsonlite::fromJSON(
      httr::content(res, "text", encoding = "UTF-8")
    )

    if (length(content) < 2 || is.null(content[[2]])) {
      message("No data returned from the World Bank API.")
      return(NULL)
    }

    data <- content[[2]]

    # CORRECCIÓN: Convertir valores vacíos a NA y asegurar tipo numérico
    value <- as.numeric(data$value)
    value[is.na(value) | value == "" | is.null(value)] <- NA

    dplyr::as_tibble(data.frame(
      indicator = data$indicator$value,
      country   = data$country$value,
      year      = as.integer(data$date),
      value     = value
    ))

  }, error = function(e) {
    message("The World Bank API is currently unavailable. Please try again later.")
    return(NULL)
  })

  return(result)
}
