#' remapResults
#'
#' This function is used to remap the results from a runPipeline run with a 
#' different order of the signatures or different names. You can either give
#' the function a new mapping of the signatures which is just a new order in 
#' which you want the signatures in. For instance, if you want to just swap the
#' first and second signatures and you have a total of 4 signatures, you would 
#' pass in c(2,1,3,4) for the sigMap parameters. The other use case is if you
#' want to rename the signatures. To do this you just have to pass a vector of 
#' names that is of the same length as the number of signatures.
#' 
#' Overall, this function will create a duplicate results folder in the same
#' directory and regenerate all of the plots and result files into the new order
#' or with the new names. This means that you don't have to regenerate all of 
#' the plots manually.
#'
#' @param path The path to the results folder to remap
#' @param sigMap The new order for the signatures
#' @param sigNames New signature names
#' @param saveRes Whether or not to save results. Default is FALSE.
#' @return no return
#' @export
remapResults = function(path, sigMap = NULL, sigNames = NULL, saveRes = FALSE){
  #Checks whether a mapping a specified
  if (is.null(sigMap) && is.null(sigNames)){
    stop(paste("Please specify either a sigMap or new sigNames."))
  }
  
  #Checks to make sure the file exists
  if (!dir.exists(path))
    stop(paste("No file found at",path," Please check for file."))
  
  #Extracts folderName and attempts to create new folder
  parentDir =  dirname(path);
  folderName = strsplit(path,"/")[[1]]
  folderName = folderName[length(folderName)]
  if (saveRes){
  	newDir = paste0(parentDir,"/",folderName,"_Remapped")
  }
  if (!dir.create(newDir)){
    stop("Failed to create new directory. Check file permissions.")
  }
  
  #Loads previous results
  if (!file.exists(paste0(path,"/","Pipeline Results.rds"))){
    stop(paste("No results file found in",path,". Make sure the file is named Pipeline Results.rds"))
  }
  prevResults = readRDS(paste0(path,"/","Pipeline Results.rds"))
  
  
  #Remaps the signatures
  newSigs = prevResults$sigs
  
  if (!is.null(sigMap))
    newSigs = newSigs[,sigMap]
  
  if (!is.null(sigNames))
    colnames(newSigs) = sigNames

  #Extract runName from folder name
  runName = sub(".*?_","",folderName)
  
  #Plots new Signatures
  plotSigs(newSigs,saveRes,newDir,runName = runName)
  
  #Replot Scm
  plotScm(t(prevResults$scm),runName,saveRes,newDir)
  
  #Resave scm
  saveRDS(prevResults$scm,file=paste0(newDir,"/",runName,"_sample_component_matrix.rds"))
  
  #Replot SigByComp
  plotSigMat(newSigs,runName,saveRes,newDir)

  if (is.null(prevResults$sigExposure))
    newExposures = t(read.table(paste0(path,"/",runName,"_sigExposureMatrix.txt"),sep = '\t',))
  else
    newExposures = prevResults$sigExposure
  
  if (!is.null(sigMap))
    newExposures = newExposures[sigMap,]
  
  if (!is.null(sigNames))
    row.names(newExposures) = sigNames
  
  #Replot Signature Exposure matrix
  plotSigExposureMat(newExposures,runName,saveRes,newDir)
  
  if (saveRes){
  	#Save the Signature Exposure Matrix
  	toWrite = newExposures
  	utils::write.table(toWrite,sep = '\t',
  	                   file=paste0(newDir,"/",runName,"_sigExposureMatrix.txt"))
  
  	#Save the Feature Contribution matrix
  	toWrite = newSigs
  	utils::write.table(toWrite, sep = '\t',
  	                   file=paste0(newDir,"/",runName,"_featureContribution.txt"))
  	
  	prevResults$sigs = newSigs;
  	prevResults$sigExposure = as.data.frame(newExposures);
  	saveRDS(prevResults,paste0(newDir,"/","Pipeline Results.rds"))
  }
} 
