% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mst.R
\name{mst}
\alias{mst}
\alias{mst.assemble}
\alias{mst.assemble.sequence}
\alias{mst.constraint}
\alias{mst.get}
\alias{mst.objective}
\alias{mst.route}
\alias{mst.sim}
\alias{mst.stagelength}
\alias{mst.summary}
\alias{plot.mst}
\alias{print.mst}
\title{Multistage Testing}
\usage{
mst(pool, design, npanel)

\method{print}{mst}(x, ...)

mst.route(x, route, op, print = FALSE)

mst.objective(x, thetas, target, routes = NULL)

mst.constraint(x, var, level, min, max, routes = NULL)

mst.stagelength(x, stage, min, max)

mst.assemble(x, ...)

mst.summary(x, ...)

\method{plot}{mst}(x, ...)

mst.get(x, panel, module.index = NULL, route.index = NULL)

mst.sim(x, theta, routing = NULL, estimator = estimate.theta.mle)

mst.assemble.sequence(x, ...)
}
\arguments{
\item{pool}{an item pool (data.frame)}

\item{design}{a vector of number of modules in stages}

\item{npanel}{the number of panels}

\item{x}{the mst object}

\item{...}{further arugments passed to lp.control or choose if summarize results \code{by.route}}

\item{route}{a vector of module index, and check \code{route} table in a mst object for routes}

\item{op}{"+" for adding a route and "-" for removing a route}

\item{print}{TRUE to print intermediate results}

\item{thetas}{a vector of theta points where information targets are set}

\item{target}{a single or vector of numeric values for targets, \code{Inf} for maximizing inforation}

\item{routes}{a vector of route index to add objective functions, or \code{NULL} for all routes}

\item{var}{the constrained variable. \code{"len"} for test length}

\item{level}{the constrained level for a categorical variable. \code{NA} or \code{NULL} for continuous variable}

\item{min}{the minimum value}

\item{max}{the maximum value}

\item{stage}{the stage being constrained}

\item{panel}{the panel index}

\item{module.index}{the module index}

\item{route.index}{the route index}

\item{theta}{the value of true theta}

\item{routing}{a list of routing cut scores or \code{NULL} for auto routing}

\item{estimator}{an ability estimation method.}
}
\value{
\code{mst} returns a \code{mst} object
}
\description{
\code{mst} creates a multistage (MST) object

\code{mst.route} adds/removes a route to/from the mst object

\code{mst.info.obj} adds information objective functions

\code{mst.constraint} sets constraints

\code{mst.stagelength} sets the min/max length for a stage

\code{mst.assemble} solves the LP object and assembles items

\code{mst.summary} summarized an assembled mst object

\code{mst.get} retreives items from an assembled mst object

\code{mst.sim} conducts a simulation for MST

\code{mst.assemble.sequence} assembles panels in sequence rather than simultaneously.
}
\details{
The \code{mst} object contains \code{pool} (item pool), \code{route} (permissilbe routes), 
\code{lp} (test assembly object), \code{items} (assembled items after the lp is solve),
\code{nstage} (number of stage), \code{nmodule} (number of modules), \code{npanel} (number of panels),
\code{nroute} (number of routes), \code{module} (module information).

In \code{mst.summary}, passing \code{by.route=TRUE} to summarize results by routes;
otherwise, by modules. A list, consisting of data and plot, is return.

In \code{plot.mst}, use \code{by.route=TRUE} to plot TIFs for routes;
otherwise, TIFs for modules.

In \code{mst.sim}, auto routing method routes a test taker to one of the permissible modules 
that has the largest information. Users can use any theta estimator from the Estimation module. 
See \code{estimate.theta.mle}, \code{estimate.theta.map}, and \code{estimate.theta.eap}
for example. When writing new estimator, make sure it takes arguments as \code{foo(u, a, b, c)}.

Use \code{mst.assemble.sequence} when the LP problem is too large to be solved in \code{mst.assemble}.
The results will ievitably favor the panels assembled earlier than later. Try to run another round
of \code{mst.assemble} using those assembled items (perhapse, plus a bit more randomly selected items)
to reduce the unparallelism between panels.
}
\examples{
# generate an item pool
pool <- gen.irt(1, 200)$items
pool$content <- sample(1:3, nrow(pool), replace=TRUE)
# 1-2-3 design with 2 parallel panels
x <- mst(pool, c(1, 2, 3), 2)
# remove route with radical changes from the 1-3-3 design
x <- mst.route(x, c(1, 2, 6), "-", print=TRUE)
x <- mst.route(x, c(1, 3, 4), "-", print=TRUE)
# maximize information at -1.0/0/1.0 for low-/mid-/high-ability routes
x <- mst.objective(x, -1.0, Inf, routes=1)
x <- mst.objective(x,  0.0, Inf, routes=2:3)
x <- mst.objective(x,  1.0, Inf, routes=4)
# all routes to have 10 items
x <- mst.constraint(x, "len", NA, 10, 10)
x <- mst.constraint(x, "content", 1, 3, 3)
# all least 2 items each stage
x <- mst.stagelength(x, 1, 2, 10)
x <- mst.stagelength(x, 2, 2, 10)
x <- mst.stagelength(x, 3, 2, 10)
# assemble
x <- mst.assemble(x)
# check content
for(p in 1:x$npanel){
 for(r in 1:x$nroute){
   cat("Panel #", p, " Route #", r, ", Content Area 1 has ",
   freq(mst.get(x, panel=p, route=r)$content, 1:3)$n[1], " items.\\n", sep="")
 }
}
# check TIFs
plot(x, by.route=TRUE)
# run simulation
mst.sim(x,  1.5)
mst.sim(x,  0.5)
mst.sim(x, -0.5)
mst.sim(x, -1.5)
# Solve a big problem in sequence
rm("x")
pool <- gen.irt(1, 1000)$items
pool$content <- sample(1:3, nrow(pool), replace=TRUE)
x <- mst(pool, c(1, 2, 3), 3)
x <- mst.route(x, c(1, 2, 6), "-")
x <- mst.route(x, c(1, 3, 4), "-")
x <- mst.objective(x, -1.0, Inf, routes=1)
x <- mst.objective(x,  0.0, Inf, routes=2:3)
x <- mst.objective(x,  1.0, Inf, routes=4)
x <- mst.constraint(x, "len", NA, 10, 10)
x <- mst.constraint(x, "content", 1, 3, 3)
x <- mst.stagelength(x, 1, 2, 10)
x <- mst.stagelength(x, 2, 2, 10)
x <- mst.stagelength(x, 3, 2, 10)
# unlikely to be solved simulatanouesly within 1 minute
# invisible(mst.assemble(x, timeout=60))
# may be solved in sequence within 1 minute
x <- mst.assemble.sequence(x, timeout=60)
plot(x, by.route=TRUE)
}

