% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/normals.R
\name{normals_dl}
\alias{normals_dl}
\title{Download climate normals from Environment and Climate Change Canada}
\usage{
normals_dl(
  climate_ids,
  normals_years = "1981-2010",
  format = TRUE,
  stn = NULL,
  verbose = FALSE,
  quiet = FALSE
)
}
\arguments{
\item{climate_ids}{Character. A vector containing the Climate ID(s) of the
station(s) you wish to download data from. See the \code{\link{stations}}
data frame or the \code{\link{stations_search}} function to find Climate
IDs.}

\item{normals_years}{Character. The year range for which you want climate
normals. Default "1981-2010".}

\item{format}{Logical. If TRUE (default) formats measurements to numeric and
date accordingly. Unlike \code{weather_dl()}, \code{normals_dl()} will always format
column headings as normals data from ECCC cannot be directly made into a
data frame without doing so.}

\item{stn}{DEFUNCT. Now use \code{stations_dl()} to update internal data and
\code{stations_meta()} to check the date it was last updated.}

\item{verbose}{Logical. Include progress messages}

\item{quiet}{Logical. Suppress all messages (including messages regarding
missing data, etc.)}
}
\value{
tibble with nested normals and first/last frost data
}
\description{
Downloads climate normals from Environment and Climate Change Canada (ECCC)
for one or more stations (defined by \code{climate_id}s). For details and units,
see the \code{\link{glossary_normals}} data frame or the \code{glossary_normals} vignette:
\code{vignette("glossary_normals", package = "weathercan")}
}
\details{
Climate normals from ECCC include two types of data, averages by
month for a variety of measurements as well as data relating to the
frost-free period. Because these two data sources are quite different, we
return them as nested data so the user can extract them as they wish. See
examples for how to use the \code{unnest()} function from the
\href{https://tidyr.tidyverse.org/}{\code{tidyr}}
package to extract the two different datasets.

The data also returns a column called \code{meets_wmo} this reflects whether or
not the climate normals for this station met the WMO standards for
temperature and precipitation (i.e. both have code >= A). Each measurement
column has a corresponding \verb{_code} column which reflects the data quality
of that measurement (see the \href{https://climate.weather.gc.ca/doc/Canadian_Climate_Normals_1981_2010_Calculation_Information.pdf}{1981-2010 ECCC calculations document}
or the \href{https://climate.weather.gc.ca/doc/Canadian_Climate_Normals_1971_2000_Calculation_Information.pdf}{1971-2000 ECCC calculations document}
for more details)

Climate normals are downloaded from the url stored in option
\code{weathercan.urls.normals}. To change this location use:
\code{options(weathercan.urls.normals = "your_new_url")}.
}
\examples{
\dontshow{if (check_eccc()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}

# Find the climate_id
stations_search("Brandon A", normals_years = "current")

# Download climate normals 1981-2010
n <- normals_dl(climate_ids = "5010480")
n

# Pull out last frost data
library(tidyr)
f <- unnest(n, frost)
f

# Pull out normals
nm <- unnest(n, normals)
nm

# Download climate normals 1971-2000
n <- normals_dl(climate_ids = "5010480", normals_years = "1971-2000")
n

# Note that some do not have last frost dates
n$frost

# Download multiple stations for 1981-2010,
n <- normals_dl(climate_ids = c("301C3D4", "301FFNJ", "301N49A"))
n

# Note, putting both into the same data set can be done but makes for
# a very unweildly dataset (there is lots of repetition)
nm <- unnest(n, normals)
f <- unnest(n, frost)
both <- dplyr::full_join(nm, f)
both
\dontshow{\}) # examplesIf}
}
