#' Visual Studio Custom API Calls
#'
#' @description
#' For any requirement not currently in place in the \code{vstsr} package, then this function will allow you
#' to use the relevant API call without any extra requirements.
#'
#' For the most part it is just a shell of \code{\link[httr]{VERB}} but will have the \code{auth_key} set up already.
#'
#' @param url the URI of the API call to run
#' @param verb name of the verb to use
#' @param auth_key authentication key generated by using \code{\link{vsts_auth_key}}
#' @param body check \code{\link[httr]{VERB}} for more details. If the object is a named list, then it will be
#' transformed into a JSON string so that can be added to the call. Use
#' \url{https://docs.microsoft.com/en-us/rest/api/vsts/} to find out any required parameter for the body.
#' @param query a list of extra parameters that can be sent to the API call. If not required then leave as \code{NULL}
#'
#' @examples
#' \dontrun{
#' auth_key <- vsts_auth_key('<username>', '<password>')
#' #Get commits of a repository
#' URL <- paste0('https://{accountName}.visualstudio.com/{project}',
#'               '/_apis/git/repositories/{repositoryId}/commits/',
#'               '{commitId}?api-version=4.1-preview')
#' vsts_run_command(URL, 'GET', auth_key)
#' }
#'
#' @export
vsts_run_command <- function(url, verb, auth_key, body = NULL, query = NULL) {

  content_body <- if(!is.null(body)) jsonlite::toJSON(body, auto_unbox = TRUE) else NULL

  response <- httr::VERB(verb = verb, url = url, config = httr::add_headers(Authorization = auth_key),
                         httr::content_type_json(), query = query, body = content_body)

  if(httr::status_code(response) >= 300) {
    cat(httr::http_condition(response, 'message', 'run custom command')$message, '\n')
    return(invisible(NULL))
  }

  content <- httr::content(response, as = 'text', encoding = 'UTF-8') %>% jsonlite::fromJSON(., flatten = TRUE)
  return(invisible(content))
}
