#' Visual Studio Project Build Definition Information
#'
#' @description
#' These functions will allow you to scrape build definition information from Visual Studio.
#'
#' @details
#' For more information about the build definition API calls check
#' \url{https://docs.microsoft.com/en-us/rest/api/vsts/build/definitions}.
#'
#' @param domain the location of the visual studio server
#' @param project the name of the project in \code{domain} to look at
#' @param auth_key authentication key generated by using \code{\link{vsts_auth_key}}
#' @param query a list of extra parameters that can be sent to the API call:
#' \describe{
#' \item{revision}{[integer] The revision number to retrieve. If this is not specified, the latest version will be returned.}
#' \item{minMetricsTime}{[string] If specified, indicates the date from which metrics should be included.}
#' \item{propertyFilters}{[string] A comma-delimited list of properties to include in the results.}
#' \item{includeLatestBuilds}{[logical]}
#' }
#'
#' @examples
#' #Add in own details to get a non-NULL output
#' auth_key <- vsts_auth_key('<username>', '<password>')
#' vsts_get_build_defs('domain', 'project', auth_key)
#'
#' @rdname vsts_build_def
#' @export
vsts_get_build_defs <- function(domain, project, auth_key, query = NULL) {
  uri <- paste0('https://', domain, '.visualstudio.com/', project, '/_apis/build/builds?api-version=4.1-preview')

  response <- httr::GET(uri, httr::add_headers(Authorization = auth_key), query = query)
  if(response$status_code != 200) {
    cat(httr::http_condition(response, 'message', 'get build definition list')$message, '\n')
    return(invisible(NULL))
  }

  content <- httr::content(response, as = 'text', encoding = 'UTF-8') %>% jsonlite::fromJSON(.) %>% .$value
  invisible(content)
}
