DEFAULT_RSPM_REPO_ID <-  "1" # cran
DEFAULT_RSPM <-  "https://packagemanager.rstudio.com"

#' Write a Dockerfile for a vetiver model
#'
#' After creating a Plumber file with [vetiver_write_plumber()], use
#' `vetiver_write_docker()` to create a Dockerfile plus a `vetiver_renv.lock`
#' file for a pinned [vetiver_model()].
#'
#' @inheritParams vetiver_api
#' @param plumber_file A path for your Plumber file, created via
#' [vetiver_write_plumber()]. Defaults to `plumber.R` in the working directory.
#' @param path A path to write the Dockerfile and `renv.lock` lockfile,
#' capturing the model's package dependencies. Defaults to the working directory.
#' @param lockfile The generated lockfile in `path`. Defaults to
#' `"vetiver_renv.lock"`.
#' @param rspm A logical to use the
#' [RStudio Public Package Manager](https://packagemanager.rstudio.com/) for
#' [renv::restore()] in the Docker image. Defaults to `TRUE`.
#' @param port The server port for listening: a number such as 8080 or an
#' expression like `'as.numeric(Sys.getenv("PORT"))'` when the port is injected
#' as an environment variable.
#' @param expose Add `EXPOSE` to the Dockerfile? This is helpful for using
#' Docker Desktop but does not work with an expression for `port`.
#'
#' @return The content of the Dockerfile, invisibly.
#' @export
#'
#' @examplesIf interactive() || identical(Sys.getenv("IN_PKGDOWN"), "true")
#'
#' library(pins)
#' tmp_plumber <- tempfile()
#' b <- board_temp(versioned = TRUE)
#' cars_lm <- lm(mpg ~ ., data = mtcars)
#' v <- vetiver_model(cars_lm, "cars_linear")
#' vetiver_pin_write(b, v)
#' vetiver_write_plumber(b, "cars_linear", file = tmp_plumber)
#'
#' ## default port
#' vetiver_write_docker(v, tmp_plumber, tempdir())
#' ## port from env variable
#' vetiver_write_docker(v, tmp_plumber, tempdir(),
#'                      port = 'as.numeric(Sys.getenv("PORT"))',
#'                      expose = FALSE)
#'
vetiver_write_docker <- function(vetiver_model,
                                 plumber_file = "plumber.R",
                                 path = ".",
                                 lockfile = "vetiver_renv.lock",
                                 rspm = TRUE,
                                 port = 8000,
                                 expose = TRUE) {

    from_r_version <- glue::glue("FROM rocker/r-ver:{getRversion()}")
    rspm_env <- ifelse(
        rspm,
        "ENV RENV_CONFIG_REPOS_OVERRIDE https://packagemanager.rstudio.com/cran/latest\n",
        ""
    )

    pkgs <- unique(c(docker_pkgs, vetiver_model$metadata$required_pkgs))
    pkgs <- setdiff(pkgs, drop_pkgs)
    renv::snapshot(
        project = path,
        lockfile = lockfile,
        packages = pkgs,
        prompt = FALSE,
        force = TRUE
    )
    plumber_file <- fs::path_rel(plumber_file)
    sys_reqs <- glue_sys_reqs(pkgs)
    copy_renv <- glue("COPY {lockfile} renv.lock")
    copy_plumber <- glue("COPY {plumber_file} /opt/ml/plumber.R")
    expose <- ifelse(expose, glue("EXPOSE {port}"), "")
    entrypoint <- glue('ENTRYPOINT ["R", "-e", ',
                       '"pr <- plumber::plumb(\'/opt/ml/plumber.R\'); ',
                       'pr$run(host = \'0.0.0.0\', port = {port})"]')


    ret <- compact(list(
        "# Generated by the vetiver package; edit with care\n",
        ## https://github.com/rstudio/plumber/blob/main/Dockerfile:
        from_r_version,
        rspm_env,
        sys_reqs,
        "",
        copy_renv,
        'RUN Rscript -e "install.packages(\'renv\')"',
        'RUN Rscript -e "renv::restore()"',
        copy_plumber,
        expose,
        entrypoint
    ))

    readr::write_lines(ret, file = file.path(path, "Dockerfile"))
}

docker_pkgs <- c("pins", "plumber", "rapidoc", "vetiver", "renv")
drop_pkgs <- "stats"

glue_sys_reqs <- function(pkgs) {
    rlang::check_installed(c("curl", "jsonlite"))
    rspm <- Sys.getenv("RSPM_ROOT", DEFAULT_RSPM)
    rspm_repo_id <- Sys.getenv("RSPM_REPO_ID", DEFAULT_RSPM_REPO_ID)
    rspm_repo_url <- glue("{rspm}/__api__/repos/{rspm_repo_id}")

    pkgnames <- glue_collapse(pkgs, sep = "&pkgname=")

    req_url <- glue(
        "{rspm_repo_url}/sysreqs?all=false",
        "&pkgname={pkgnames}&distribution=ubuntu&release=20.04"
    )
    res <- curl::curl_fetch_memory(req_url)
    sys_reqs <- jsonlite::fromJSON(rawToChar(res$content), simplifyVector = FALSE)
    if (!is.null(sys_reqs$error)) {
        rlang::abort(sys_reqs$error)
    }
    sys_reqs <- map(sys_reqs$requirements, pluck, "requirements", "packages")
    sys_reqs <- sort(unique(unlist(sys_reqs)))
    sys_reqs <- glue_collapse(sys_reqs, sep = " \\\n  ")
    glue(
        "RUN apt-get update -qq && ",
        "apt-get install -y --no-install-recommends \\\n  ",
        sys_reqs,
        " \\\n  && apt-get clean",
        .trim = FALSE
    )
}

