\name{turf}
\alias{turf}
\title{TURF Analysis for R}
\description{Calculate weighted reach and frequency statistics from TURF data for all possible combinations of \emph{n} choose 
  \emph{k}, for user-specified combination sets, or for Monte Carlo simulated subsets of combinations. 
}

\usage{
turf(data, n, k, combos, ...)
}

\arguments{
  \item{data}{Required. Literal character string representing name of a file in the working directory readable using 
    \code{read.table(data, header=TRUE)}, or name of a data frame or matrix in R containing TURF data. Rows are
	individuals (respondents). Columns are (1) respondent identifier, (2) a weight variable, and a minimum of \emph{n} columns 
	containing only zeroes and ones, each representing an individual item in the TURF algorithm. Respondent identifiers need not 
	be unique and weights need not sum to the total number of rows. In the absence of any weight variable, substitute a column of
    ones. Ones in the remaining columns indicate that the reach criterion was met for a given item by a given individual. Values 
	other than zero or one in these columns (including \code{NA}) trigger an error. \code{data} may contain more than \emph{n} + 2 
	columns, but any columns in addition to that number will be ignored.
  }
  \item{n}{Required. Scalar indicating the number of items to be included in the TURF algorithm; 0 < \emph{n} < (\code{ncol(data)} - 1).
    Non-integer values are coerced using \code{floor(}\emph{n}\code{)}.
  }
  \item{k}{Required. Vector of length 1 to \emph{n} containing any values 1 to \emph{n} indicating the combination sizes to be evaluated 
    by the TURF algorithm. Non-integer values are coerced using \code{floor(}\emph{k}\code{)}.
  }
  \item{combos}{Optional. List of combination sets to be evaluated by the TURF algorithm, such as that generated by \code{turf.combos}. 
    Individual combination sets are \emph{p} x \emph{n} matrices, containing only zeroes and ones indicating items to be included in a 
	given combination. Rows (\emph{p}) correspond to combinations evaluated; columns (\emph{n}) correspond to items, and must be in the 
	same order as the items columns in \code{data}. Each \emph{i}th element of \code{combos} should contain combinations of the size 
	specified in the \emph{i}th position of \emph{k}; \code{length(combos)} must be equal to \code{length(}\emph{k}\code{)}. See "details" 
	for additional information on usage.	
  }
  \item{...}{Optional. Additional arguments controlling behavior and output of the TURF algorithm. See \code{turf.args}. Arguments indicated
    here must match named arguments in \code{turf.args}.  
  }
}

\details{
  TURF algorithm is as originally described by Miaoulis, \emph{et al}. (1990) and outputs \emph{reach} and \emph{frequency} statistics 
  for each combination evaluated.
  
  Reach = sum(weights[reached]) / sum(weights), where an individual is considered "reached" if the reach criterion is met for at least
  the number of items indicated by the \code{depth} argument in \code{turf.args}. See Markowitz (2005) for a more detailed explanation
  of \emph{depth of reach}.  
  
  Frequency = sum(weights \code{x} items_reached) / sum(weights). Frequency includes all individuals, whether reached or not. Frequency 
  among "reached" individuals may be calculated as Frequency / Reach regardless of weights.
  
  When the \code{combos} argument is omitted from the call, combination sets are generated automatically using \code{turf.combos} and 
  information passed to it by \emph{n}, \emph{k} and any additional arguments passed from \code{turf.args}. The user may also pass any
  combination sets (e.g., a user-truncated set) developed in or out of R, so long as the structure of the combination sets corresponds to 
  that described above.
  
  Monte Carlo simulated combination sets, when requested, are generated using the procedure described by Adler, \emph{et al}. (2010). Sets of
  \emph{n} choose \emph{k} that result in small numbers of combinations should be run without the Monte Carlo simulation option since they
  require little processing time (e.g., 12 choose 7 = 792 combinations and will run in less than 1 second). Typically, even larger sets will run
  in a reasonable amount of time without subsetting (e.g., 18 choose 10 = 43,758 combinations and will run in less than a minute on most 
  computers), but RAM may become a limiting factor, especially when large numbers of individuals are combined with problems involving large
  numbers of items. The default for substituting Monte Carlo simulated subsets of combinations is set at 10,000. For the above reasons, 
  and because subsetting can lead to duplicate combinations especially when the original set is small, care should be taken to not subset 
  combinations until the original set becomes unmanageably large.
  
  The lack of looping in the algorithm permits processing at about 1,000 combinations per second.
} 

\value{
  R object:
  
  A list of 2 elements
  \item{turf}{A list of \code{length(}\emph{k}\code{)} elements, each of which is comprised of a matrix whose columns are \emph{reach}
    (\code{rchX}), \emph{frequency} (\code{frqX}), and \emph{n} indicator variables representing the items included in a given combination. 
	Rows represent combinations evaluated. Sorting and the number of rows returned are controlled by the \code{sort} and \code{keep} arguments 
	passed from \code{turf.args}.
  }
  \item{call}{The call to \code{turf} as a literal character string
  }\cr
  Written to Console:
  
  Combination size(s) evaluated, as \emph{k} of \emph{n}; time in seconds required to evaluate all combinations of a given size;
  total time in seconds required to complete the function call.
}

\references{
  Adler, T.J., Smith, C. & Dumont, J. 2010. Optimizing product portfolios using discrete choice modeling and TURF. In: S. Hess,
  A. Daly (Eds), \emph{Choice modeling: the state-of-the-art and the state-of-practice; proceedings from the Inaugural 
  International Choice Modeling Conference}. Emerald Publishing Group Ltd., pp. 485-497.
  
  Krieger, A.M. & Green, P.E. 2000. Turf revisited: Enhancements to toal unduplicated reach and frequency analysis. \emph{Marketing
  Research}, 12, 30-36.
  
  Markowitz, L. 2005. Going beyond TURF to complement and extend existing product lines. \emph{Ipsos-Insight}, November 2005.
  
  Miaoulis, G., Free, V. & Parsons, H. 1990. TURF: A new planning approach for product line extensions. \emph{Marketing Research},
  March, pp. 28-40.
}

\examples{
  ##Example 1
  ##Evaluate all combinations of 3, 4, 5 and 6 items drawn from 10 items.
  data(turf_ex_data)
  ex1 <- turf(turf_ex_data, 10, 3:6)
  
  ##Output to Console:
  ##3 of 10: 0.105068 sec
  ##4 of 10: 0.1420949 sec
  ##5 of 10: 0.1511021 sec
  ##6 of 10: 0.1160791 sec
  ##total time elapsed: 0.518347 sec
  
  
  ##Example 2
  ##Pass additional arguments 
  data(turf_ex_data)
  ex2 <- turf(turf_ex_data, 10, 3:6, depth=2, keep=20, mc=TRUE, nsims=100) 
  
  ##Output to Console:
  ##3 of 10: 0.03802586 sec
  ##4 of 10: 0.03702521 sec
  ##5 of 10: 0.0380249 sec
  ##6 of 10: 0.03802609 sec
  ##total time elapsed: 0.156105 sec 
  
  
  ##Example 3
  ##Customize combos, exclude item 10 from all combinations of size 3
  data(turf_ex_data)
  psims <- colSums(turf_ex_data[,-c(1:2)]) / sum(turf_ex_data[,2])
  psims <- psims / sum(psims)
  combos <- turf.combos(10, 3, mc=TRUE, nsims=100, psims=psims)
  combos[[1]] <- combos[[1]][-which(combos[[1]][,10]==1),]
  ex3 <- turf(turf_ex_data, 10, 3, combos)
  
  ##Output to Console:
  ##3 of 10: 0.02001405 sec
  ##total time elapsed: 0.02001405 sec 
  
}