% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rpsftm.R
\name{rpsftm}
\alias{rpsftm}
\title{Rank Preserving Structural Failure Time Model (RPSFTM) for
Treatment Switching}
\usage{
rpsftm(
  data,
  stratum = "",
  time = "time",
  event = "event",
  treat = "treat",
  rx = "rx",
  censor_time = "censor_time",
  base_cov = "",
  low_psi = -1,
  hi_psi = 1,
  n_eval_z = 100,
  treat_modifier = 1,
  recensor = TRUE,
  admin_recensor_only = FALSE,
  autoswitch = TRUE,
  gridsearch = FALSE,
  alpha = 0.05,
  ties = "efron",
  tol = 1e-06,
  boot = FALSE,
  n_boot = 1000,
  seed = NA
)
}
\arguments{
\item{data}{The input data frame that contains the following variables:
\itemize{
\item \code{stratum}: The stratum.
\item \code{time}: The survival time for right censored data.
\item \code{event}: The event indicator, 1=event, 0=no event.
\item \code{treat}: The randomized treatment indicator, 1=treatment,
0=control.
\item \code{rx}: The proportion of time on active treatment.
\item \code{censor_time}: The administrative censoring time. It should
be provided for all subjects including those who had events.
\item \code{base_cov}: The baseline covariates (excluding treat).
}}

\item{stratum}{The name(s) of the stratum variable(s) in the input data.}

\item{time}{The name of the time variable in the input data.}

\item{event}{The name of the event variable in the input data.}

\item{treat}{The name of the treatment variable in the input data.}

\item{rx}{The name of the rx variable in the input data.}

\item{censor_time}{The name of the censor_time variable in the input data.}

\item{base_cov}{The vector of names of baseline covariates (excluding
treat) in the input data.}

\item{low_psi}{The lower limit of the causal parameter.}

\item{hi_psi}{The upper limit of the causal parameter.}

\item{n_eval_z}{The number of points between low_psi and hi_psi
(inclusive) at which to evaluate the log-rank Z-statistics.}

\item{treat_modifier}{The optional sensitivity parameter for the
constant treatment effect assumption.}

\item{recensor}{Whether to apply recensoring to counter-factual
survival times. Defaults to \code{TRUE}.}

\item{admin_recensor_only}{Whether to apply recensoring to administrative
censoring time only. Defaults to \code{FALSE}, in which case,
recensoring will be applied to the actual censoring time for dropouts.}

\item{autoswitch}{Whether to exclude recensoring for treatment arms
with no switching. Defaults to \code{TRUE}.}

\item{gridsearch}{Whether to use grid search to estimate the causal
parameter \code{psi}. Defaults to \code{FALSE}, in which case, a root
finding algorithm will be used.}

\item{alpha}{The significance level to calculate confidence intervals.}

\item{ties}{The method for handling ties in the Cox model, either
"breslow" or "efron" (default).}

\item{tol}{The desired accuracy (convergence tolerance) for \code{psi}.}

\item{boot}{Whether to use bootstrap to obtain the confidence
interval for hazard ratio. Defaults to \code{FALSE}, in which case,
the confidence interval will be constructed to match the log-rank
test p-value.}

\item{n_boot}{The number of bootstrap samples.}

\item{seed}{The seed to reproduce the bootstrap results.
The seed from the environment will be used if left unspecified.}
}
\value{
A list with the following components:
\itemize{
\item \code{psi}: The estimated causal parameter.
\item \code{psi_CI}: The confidence interval for \code{psi}.
\item \code{psi_CI_type}: The type of confidence interval for \code{psi},
i.e., "grid search", "root finding", or "bootstrap".
\item \code{Sstar}: A data frame containing the counter-factual untreated
survival times and the event indicators.
\item \code{kmstar}: A data frame containing the Kaplan-Meier estimates
based on the counter-factual untreated survival times by treatment arm.
\item \code{eval_z}: A data frame containing the log-rank test Z-statistics
evaluated at a sequence of \code{psi} values. Used to plot and check
if the range of \code{psi} values to search for the solution and
limits of confidence interval of \code{psi} need be modified.
\item \code{logrank_pvalue}: The two-sided p-value of the log-rank test
based on the treatment policy strategy.
\item \code{cox_pvalue}: The two-sided p-value for treatment effect based on
the Cox model.
\item \code{hr}: The estimated hazard ratio from the Cox model.
\item \code{hr_CI}: The confidence interval for hazard ratio.
\item \code{hr_CI_type}: The type of confidence interval for hazard ratio,
either "log-rank p-value" or "bootstrap".
\item \code{settings}: A list with the following components:
\itemize{
\item \code{low_psi}: The lower limit of the causal parameter.
\item \code{hi_psi}: The upper limit of the causal parameter.
\item \code{n_eval_z}: The number of points between \code{low_psi} and
\code{hi_psi} at which to evaluate the log-rank Z-statistics.
\item \code{treat_modifier}: The sensitivity parameter for the
constant treatment effect assumption.
\item \code{recensor}: Whether to apply recensoring to counter-factual
survival times.
\item \code{admin_recensor_only}: Whether to apply recensoring to
administrative censoring time only.
\item \code{autoswitch}: Whether to exclude recensoring for treatment
arms with no switching.
\item \code{gridsearch}: Whether to use grid search to estimate the
causal parameter \code{psi}.
\item \code{alpha}: The significance level to calculate confidence
intervals.
\item \code{ties}: The method for handling ties in the Cox model.
\item \code{tol}: The desired accuracy (convergence tolerance).
\item \code{boot}: Whether to use bootstrap to obtain the confidence
interval for hazard ratio.
\item \code{n_boot}: The number of bootstrap samples.
\item \code{seed}: The seed to reproduce the simulation results.
}
\item \code{hr_boots}: The bootstrap hazard ratio estimates if \code{boot} is
\code{TRUE}.
\item \code{psi_boots}: The bootstrap \code{psi} estimates if \code{boot} is
\code{TRUE}.
}
}
\description{
Obtains the causal parameter estimate from the
log-rank test and the hazard ratio estimate from the Cox model
to account for treatment switching.
}
\details{
We use the following steps to obtain the hazard ratio estimate
and confidence interval had there been no treatment switching:
\itemize{
\item Use RPSFTM to estimate the causal parameter \eqn{\psi} based on the
log-rank test for counter-factual untreated survival times for
both arms: \eqn{U_{i,\psi} = T_{C_i} +  e^{\psi}T_{E_i}}.
\item Fit the Cox proportional hazards model to the observed survival times
on the treatment arm and the counter-factual untreated survival times
on the control arm to obtain the hazard ratio estimate.
\item Use either the log-rank test p-value for the treatment policy strategy
or bootstrap to construct the confidence interval for hazard ratio.
}
}
\examples{

library(dplyr)

# Example 1: one-way treatment switching (control to active)

data <- immdef \%>\% mutate(rx = 1-xoyrs/progyrs)

fit1 <- rpsftm(
  data, time = "progyrs", event = "prog", treat = "imm",
  rx = "rx", censor_time = "censyrs", boot = FALSE)

c(fit1$hr, fit1$hr_CI)

# Example 2: two-way treatment switching (illustration only)

# the eventual survival time
shilong1 <- shilong \%>\%
  arrange(bras.f, id, tstop) \%>\%
  group_by(bras.f, id) \%>\%
  filter(row_number() == n()) \%>\%
  select(-c("ps", "ttc", "tran"))

shilong2 <- shilong1 \%>\%
  mutate(rx = ifelse(co, ifelse(bras.f == "MTA", dco/ady, 
                                1 - dco/ady),
                     ifelse(bras.f == "MTA", 1, 0)))

fit2 <- rpsftm(
  shilong2, time = "tstop", event = "event",
  treat = "bras.f", rx = "rx", censor_time = "dcut",
  base_cov = c("agerand", "sex.f", "tt_Lnum", "rmh_alea.c",
               "pathway.f"),
  low_psi = -3, hi_psi = 3, boot = FALSE)

c(fit2$hr, fit2$hr_CI)

}
\references{
James M. Robins and Anastasios A. Tsiatis.
Correcting for non-compliance in randomized trials using rank preserving
structural failure time models.
Communications in Statistics. 1991;20(8):2609-2631.

Ian R. White, Adbel G. Babiker, Sarah Walker, and Janet H. Darbyshire.
Randomization-based methods for correcting for treatment changes:
Examples from the CONCORDE trial.
Statistics in Medicine. 1999;18:2617-2634.
}
\author{
Kaifeng Lu, \email{kaifenglu@gmail.com}
}
