% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vec-smooth.R
\name{smooth_vec}
\alias{smooth_vec}
\title{Smoothing Transformation using Loess}
\usage{
smooth_vec(x, period = 30, span = NULL, degree = 2)
}
\arguments{
\item{x}{A numeric vector to have a smoothing transformation applied.}

\item{period}{The number of periods to include in the local smoothing.
Similar to window size for a moving average.
See details for an explanation \code{period} vs \code{span} specification.}

\item{span}{The span is a percentage of data to be included
in the smoothing window. Period is preferred for shorter windows
to fix the window size.
See details for an explanation \code{period} vs \code{span} specification.}

\item{degree}{The degree of the polynomials to be used.
Accetable values (least to most flexible): 0, 1, 2.
Set to 2 by default for 2nd order polynomial (most flexible).}
}
\value{
A numeric vector
}
\description{
\code{smooth_vec()} applies a LOESS transformation to a numeric vector.
}
\details{
\strong{Benefits:}
\itemize{
\item When using \code{period}, the effect is
\strong{similar to a moving average without creating missing values.}
\item When using \code{span}, the effect is to detect the trend in a series
\strong{using a percentage of the total number of observations.}
}

\strong{Loess Smoother Algorithm}
This function is a simplified wrapper for the \code{stats::loess()}
with a modification to set a fixed \code{period} rather than a percentage of
data points via a \code{span}.

\strong{Why Period vs Span?}
The \code{period} is fixed whereas the \code{span} changes as the number of observations change.

\strong{When to use Period?}
The effect of using a \code{period} is similar to a Moving Average where the Window Size
is the \emph{\strong{Fixed Period}}. This helps when you are trying to smooth local trends.
If you want a 30-day moving average, specify \code{period = 30}.

\strong{When to use Span?}
Span is easier to specify when you want a \emph{\strong{Long-Term Trendline}} where the
window size is unknown. You can specify \code{span = 0.75} to locally regress
using a window of 75\% of the data.
}
\examples{
library(dplyr)
library(ggplot2)

# Training Data
FB_tbl <- FANG \%>\%
    filter(symbol == "FB") \%>\%
    select(symbol, date, adjusted)

# ---- PERIOD ----

FB_tbl \%>\%
    mutate(adjusted_30 = smooth_vec(adjusted, period = 30, degree = 2)) \%>\%
    ggplot(aes(date, adjusted)) +
    geom_line() +
    geom_line(aes(y = adjusted_30), color = "red")

# ---- SPAN ----

FB_tbl \%>\%
    mutate(adjusted_30 = smooth_vec(adjusted, span = 0.75, degree = 2)) \%>\%
    ggplot(aes(date, adjusted)) +
    geom_line() +
    geom_line(aes(y = adjusted_30), color = "red")

# ---- Loess vs Moving Average ----
# - Loess: Using `degree = 0` to make less flexible. Comperable to a moving average.

FB_tbl \%>\%
    mutate(
        adjusted_loess_30 = smooth_vec(adjusted, period = 30, degree = 0),
        adjusted_ma_30    = slidify_vec(adjusted, .period = 30,
                                        .f = mean, .partial = TRUE)
    ) \%>\%
    ggplot(aes(date, adjusted)) +
    geom_line() +
    geom_line(aes(y = adjusted_loess_30), color = "red") +
    geom_line(aes(y = adjusted_ma_30), color = "blue") +
    labs(title = "Loess vs Moving Average")



}
\seealso{
Loess Modeling Functions:
\itemize{
\item \code{\link[=step_smooth]{step_smooth()}} - Recipe for \code{tidymodels} workflow
}

Additional Vector Functions:
\itemize{
\item Box Cox Transformation: \code{\link[=box_cox_vec]{box_cox_vec()}}
\item Lag Transformation: \code{\link[=lag_vec]{lag_vec()}}
\item Differencing Transformation: \code{\link[=diff_vec]{diff_vec()}}
\item Rolling Window Transformation: \code{\link[=slidify_vec]{slidify_vec()}}
\item Loess Smoothing Transformation: \code{\link[=smooth_vec]{smooth_vec()}}
\item Fourier Series: \code{\link[=fourier_vec]{fourier_vec()}}
\item Missing Value Imputation for Time Series: \code{\link[=ts_impute_vec]{ts_impute_vec()}}
}
}
