% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/parameter_stability.R
\name{stability}
\alias{stability}
\title{Generalized Pareto parameter estimate stability}
\usage{
stability(data, u_vec, prof = FALSE, conf = 95, mult = 1:2,
  plot_prof = FALSE, ...)
}
\arguments{
\item{data}{A numeric vector of observations.}

\item{u_vec}{A numeric vector of thresholds to be applied to the data.
Any duplicated values will be removed.  These could be set at sample
quantiles of \code{data} using \code{\link[stats]{quantile}}.}

\item{prof}{A logical scalar.  Whether to calculate confidence intervals
for the GP shape parameter \eqn{\xi} based on the profile-likelihood
for \eqn{\xi} or using the MLE plus or minus a multiple of the estimated
standard error (SE) of the MLE.  The intervals produced by the former
may be better but they take longer to calculate.
Default: \code{FALSE}.}

\item{conf}{A numeric scalar in (0, 100).  Confidence level for the
confidence intervals.  Default: 95\%.}

\item{mult}{A numeric vector of length 2.  The range of values over
which the profile log-likelihood for \eqn{\xi} is calculated is
(MLE - \code{mult[1]} c SE, MLE + \code{mult[2]} c SE),
where MLE and SE are the MLE and estimated standard error of \eqn{\xi}
and c is the constant for which this interval gives
an approximate 100\code{conf}\% level confidence interval for \eqn{\xi}
when \code{mult = c(1, 1)}.  The default, \code{mult = c(1, 2)}, works
well in most cases.  If the routine fails because the range of \eqn{\xi}
is not sufficiently wide then the relevant components of \code{mult}
should be increased.}

\item{plot_prof}{A logical scalar.  Only relevant if \code{prof = TRUE}.
If \code{plot_prof = TRUE} then the profile log-likelihood is plotted
for each threshold.  If \code{FALSE} then nothing is plotted.}

\item{...}{Further (optional) arguments to be passed to the
\code{\link[stats]{optim}} function for the optimizations on which
the profile-likelihood for \eqn{xi} is based.}
}
\value{
An object (list) of class "stability" with components:
    \item{ests}{MLEs of the GP shape parameter \eqn{\xi}.}
    \item{ses}{Estimated SEs of the MLEs of \eqn{\xi}.}
    \item{lower}{Lower limit of 100\code{conf}\% confidence intervals
     for \eqn{\xi}.}
    \item{upper}{Upper limit of 100\code{conf}\% confidence intervals
     for \eqn{\xi}.}
    \item{nexc}{The number of threshold excesses.}
    \item{u_vec}{The thresholds supplied by the user.}
    \item{u_ps}{The approximate sample quantiles to which the thresholds
     in \code{u_vec} correspond.}
    \item{data}{The input \code{data}.}
    \item{conf}{The input \code{conf}.}
  Each of these components is a numeric vector of length
\code{length(u_vec)}.
}
\description{
Uses maximum likelihood estimation to fit a Generalized Pareto (GP)
model to threshold excesses over a range of thresholds.
The threshold excesses are treated as independent and identically
distributed (i.i.d.) observations.
The resulting estimates and confidence intervals can be plotted,
using \code{\link{plot.stability}},
to produce a crude graphical diagnostic for threshold choice.
}
\details{
For each threshold in \code{u_vec} a GP model is fitted by maximum
  likelihood estimation to the threshold excesses, i.e. the amounts
  by which the data exceed that threshold.  The MLEs of the GP shape
  parameter $\eqn{\xi}$ and approximate \code{conf}\% confidence intervals
  for \eqn{\xi} are stored for plotting (by \code{\link{plot.stability}})
  to produce a simple graphical diagnostic to inform threshold selection.
  This plot is used to choose a threshold above which the underlying GP
  shape parameter may be approximately constant. See Chapter 4 of
  Coles (2001).  See also the vignette "Introducing threshr".
}
\examples{
# Set a vector of thresholds
u_vec_gom <- quantile(gom, probs = seq(0, 0.95, by = 0.05))

# Symmetric confidence intervals
gom_stab <- stability(data = gom, u_vec = u_vec_gom)
plot(gom_stab)

# Profile-likelihood-based confidence intervals
gom_stab <- stability(data = gom, u_vec = u_vec_gom, prof = TRUE)
plot(gom_stab)
}
\references{
Coles, S. G. (2001) \emph{An Introduction to Statistical
  Modeling of Extreme Values}, Springer-Verlag, London.
  \url{http://dx.doi.org/10.1007/978-1-4471-3675-0_3}
}
\seealso{
\code{\link{ithresh}} for threshold selection in the i.i.d. case
  based on leave-one-out cross-validation.

\code{\link{plot.stability}} for the S3 \code{plot} method for
  objects of class \code{stability}.

\code{\link[stats]{quantile}}.
}
