\name{quad.table}
\alias{quad.table}
\alias{quad.plot}

\title{Create and plot ``quads'' of samples}
\description{
    Pull out a ``quad'' of samples from a larger data frame, then plot the pairs of
    samples in the quad against one another.
}

\usage{
    quad.table(otu, control.before, control.after, treatment.before, treatment.after)
    quad.plot(quad)
}

\arguments{
  \item{otu}{data frame of a z- or F-transformed OTU table}
  \item{control.before}{name of the sample (in the OTU table) from the control 
    unit before the treatment is applied to the treatment unit}
  \item{control.after}{sample from the control unit after treatment is applied}
  \item{treatment.before}{sample from the treatment unit before the treatment is applied}
  \item{treatment.after}{sample from the treatment unit after the treatment is applied}
  \item{quad}{a quad generated by \code{quad.table}}
}

\details{
\code{texmexseq} was designed to compare four samples at a time: two are ``control'' samples
(before and after the treatment was applied to the ``treatment'' samples), the other two are
the treatment samples.

\code{quad.table} will grab the columns with the four given names and make them into a new
data frame (with OTU IDs kept as a separate column). This object can be plugged into \code{quad.plot}
for viewing. \code{quad.plot} expects that the OTU table will be z- or F-transformed.
}

\value{
    \item{quad.table}{returns a data frame}
    \item{quad.plot}{returns a ggplot object}
}

\seealso{
    \code{\link{z.transform.table}}
    \code{\link{f.transform.table}}
}

\examples{
# make up some data
sim.data <- function() rpoilog(1000, 1.0, 1.0, condS=TRUE)
otu <- data.frame(sample0=sim.data())
for (i in 1:10) otu[[paste('sample', i, sep='')]] <- sim.data()
otu.ids <- paste('otu', seq(1:1000), sep='')
rownames(otu) <- otu.ids
z.table <- z.transform.table(otu)

# pull out a quad, imagining that samples 1 and 2 were the control samples
# and 3 and 4 were the treatment
q <- quad.table(z.table, 'sample1', 'sample2', 'sample3', 'sample4')

# plot it
p <- quad.plot(q)
p

# ok, it's just a blob because we generated the data, but imagine we
# were particularly interested in OTUs that bloomed in the treatment
# but not in the control
interesting.otus <- filter(q, d.treatment > 2, d.control < 0)

# we can plot those in a different color
p + geom_point(data=interesting.otus, color='red')

# or see what their names are
head(arrange(interesting.otus, desc(d.treatment)))
}

\author{Scott Olesen \email{swo@mit.edu}}

\keyword{hplot}
\keyword{manip}
\keyword{design}
