% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tcata.R
\encoding{UTF-8}
\name{citation.counts}
\alias{citation.counts}
\title{Counts TCATA Citations and Observations for a Product and a Comparison Set}
\usage{
citation.counts(x, product.name = "", product.col = 1,
attribute.col = 2, results.col = NULL, comparison = "average")
}
\arguments{
\item{x}{matrix of TCATA 0/1 data with (Assessors x Products x Reps x Attributes) in rows with row headers and (Times) in columns}

\item{product.name}{name of the product for which to calculate how many times a product was checked and not checked}

\item{product.col}{index of column in \code{x} that contains the product identities}

\item{attribute.col}{index of column in \code{x} that contains the attribute identities}

\item{results.col}{indices of columns in \code{x} that contain the raw (0/1) data}

\item{comparison}{specifies whether the comparison will be with the average of \emph{all} products (\code{"average"} (default)) or with the average of the \emph{other} products (\code{"other"}, i.e. excludes the product specified by \code{product.name})}
}
\value{
list object with three elements:
\itemize{
\item{\code{P1} matrix of counts for product specified by \code{product.name}
(attributes are in rows; times are in columns).}
\item{\code{Pn} number of observations for \code{product.name}}
\item{\code{C1} matrix of counts for comparison set specified by \code{comparison}
(dimensions equal to \code{P1}.}
\item{\code{Cn} number of observations for the comparison set defined by \code{comparison}}
\item{\code{ref} a matrix of citation proportions for the comparison set specified
by \code{comparison} (dimensions equal to \code{P1}; can be used to draw a reference line;
see \code{\link[tempR]{tcata.line.plot}}}
\item{\code{declutter} a matrix for decluttering in a line plot
(dimensions equal to \code{P1}; see \code{\link[tempR]{get.decluttered}}}}
}
\description{
Calculates how many times a specified product was checked and how many times a comparison set was checked.
The number of evaluations for the product and comparison set are also calculated,
along with a reference and decluttering matrix for plotting in \code{\link[tempR]{tcata.line.plot}}.
}
\examples{
# example using 'ojtcata' data set
data(ojtcata)

# comparison of Orange Juice 3 vs. all other OJs (1, 2, 4, 5, 6)
oj3.v.other <- citation.counts(ojtcata, product.name = "3", product.col = 2,
       attribute.col = 4, results.col = 5:25, comparison = "other")

# show results
oj3.v.other

times <- get.times(colnames(ojtcata)[-c(1:4)])
attributes <- unique(ojtcata$attribute)
palettes <- make.palettes(length(attributes))

# plot results
tcata.line.plot(oj3.v.other$P1, n = oj3.v.other$Pn,
   attributes = attributes, times = times,
   line.col = palettes$pal, reference = oj3.v.other$ref, ref.lty = 3,
   declutter = oj3.v.other$declutter, highlight = TRUE, highlight.lwd = 4,
   highlight.col = palettes$pal.light,
   height = 7, width = 11, legend.cex = 0.7, main = "Product 3 vs. Other Products")
}
\references{
Castura, J.C., Antúnez, L., Giménez, A., Ares, G. (2016). Temporal check-all-that-apply (TCATA): A novel temporal sensory method for characterizing products. \emph{Food Quality and Preference}, 47, 79-90. \doi{10.1016/j.foodqual.2015.06.017}

Meyners, M., Castura, J.C. (2018). The analysis of temporal check-all-that-apply (TCATA) data. \emph{Food Quality and Preference}, 67, 67-76. \doi{10.1016/j.foodqual.2017.02.003}
}
\seealso{
\code{\link[tempR]{tcata.line.plot}}, \code{\link[tempR]{get.decluttered}}
}
