#' @title Set up package dependencies for compatibility with `renv`
#' @export
#' @seealso <https://rstudio.github.io/renv/articles/renv.html>
#' @description Write package dependencies to a script file
#'   (by default, named `_packages.R` in the root project directory).
#'   Each package is written to a separate line
#'   as a standard [library()] call (e.g. `library(package)`) so
#'   `renv` can identify them automatically.
#' @details This function gets called for its side-effect, which writes
#'   package dependencies to a script for compatibility with `renv`.
#'   The generated file should __not__ be edited by hand and will be
#'   overwritten each time `tar_renv()` is called.
#'
#'   The behavior of `renv` is to create and manage a project-local `R` library
#'   and keep a record of project dependencies in a file called `renv.lock`.
#'   To identify dependencies, `renv` crawls through code to find packages
#'   explicitly mentioned using `library()`, `require()`, or `::`.
#'   However, `targets` manages packages in a way that hides dependencies
#'   from `renv.` `tar_renv()` finds package dependencies that would be
#'   otherwise hidden to `renv` because they are declared using the `targets`
#'   API. Thus, calling `tar_renv` this is only necessary if using
#'   [`tar_option_set()`] or [`tar_target()`] to use specialized storage
#'   formats or manage packages.
#'
#'   With the script written by `tar_renv()`, `renv` is able to crawl the
#'   file to identify package dependencies (with `renv::dependencies()`).
#'   `tar_renv()` only serves to make your `targets` project compatible with
#'   `renv`, it is still the users responsibility to call `renv::init()` and
#'   `renv::snapshot()` directly to initialize and manage a
#'   project-local `R` library. This allows your `targets` pipeline to have
#'   its own self-contained `R` library separate from your standard `R`
#'   library. See <https://rstudio.github.io/renv/index.html> for
#'   more information.
#' @return Nothing, invisibly.
#' @inheritParams tar_validate
#' @param extras Character vector of additional packages to declare as
#'   project dependencies.
#' @param path Character of length 1, path to the script file to
#'   populate with `library()` calls.
#' @examples
#' tar_dir({ # Write all files to a temporary directory.
#'   tar_script({
#'     tar_option_set(packages = c("tibble", "qs"))
#'     list()
#'   })
#'   tar_renv()
#'   writeLines(readLines("_packages.R"))
#' })
#' tar_option_reset()
tar_renv <- function(
  extras = c("clustermq", "future", "rstudioapi", "visNetwork"),
  path = "_packages.R",
  callr_function = callr::r,
  callr_arguments = list()
) {
  assert_target_script()
  assert_chr(extras, "extras must be a character vector")
  assert_chr(path, "path must have type character")
  assert_scalar(path, "path must have length 1")
  assert_callr_function(callr_function)
  assert_list(callr_arguments, "callr_arguments must be a list.")
  callr_outer(
    targets_function = tar_renv_inner,
    targets_arguments = list(extras = extras, path = path),
    callr_function = callr_function,
    callr_arguments = callr_arguments
  )
  invisible()
}

tar_renv_inner <- function(pipeline, extras, path) {
  option_pkgs <- tar_option_get("packages")
  targets_pkgs <- pipeline_get_packages(pipeline)
  option_format <- store_get_packages(store_init(tar_option_get("format")))
  all_pkgs <- unique(
    c(
      option_pkgs,
      targets_pkgs,
      option_format,
      extras
    )
  )
  all_pkgs <- sort(all_pkgs)
  lines <- c(
    "# Generated by targets::tar_renv(): do not edit by hand",
    paste0("library(", all_pkgs, ")")
  )
  writeLines(lines, con = path)
}
