% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/symmetry_test.R
\name{symmetry_test}
\alias{symmetry_test}
\alias{symmetry_test.default}
\alias{symmetry_test.lm}
\alias{symmetry_test.fGARCH}
\title{Perform symmetry tests}
\usage{
symmetry_test(x, ...)

\method{symmetry_test}{default}(x, stat, mu = NULL, bootstrap = TRUE,
  B = 1000, boot_method = c("sign", "reflect"), trim = 0, k = 0,
  ...)

\method{symmetry_test}{lm}(x, stat, B = 1000, boot_method = c("sign",
  "reflect"), k = 0, ...)

\method{symmetry_test}{fGARCH}(x, stat, B = 1000, burn = 0,
  boot_method = c("sign", "reflect"), k = 0, approximate = FALSE,
  ...)
}
\arguments{
\item{x}{an object of class numeric, lm or fGARCH}

\item{...}{not used}

\item{stat}{a character vector indicating the test statistic to be used (see
\link[=TestStatistics]{Available Test Statistics})}

\item{mu}{the centre parameter around which to test symmetry}

\item{bootstrap}{a logical indicating whether to use bootstrap}

\item{B}{the number of bootstrap replications}

\item{boot_method}{the method of bootstrap sample generation (see Details)}

\item{trim}{the trim value used for estimating the centre (as used in "mean")}

\item{k}{the k parameter of the statistic, ignored if the test statistic
doesn't depend on a parameter (see \link[=TestStatistics]{Test Statistics})}

\item{burn}{the number of elements to remove from the beginning of the time
series for testing}

\item{approximate}{a logical indicating whether to use the faster approximate
bootstrap method (see Details)}
}
\value{
An object of class "htest" containing the results of the testing.
}
\description{
This is a generic function used to perform symmetry tests on numeric vectors
or objects of class lm (linear models) and objects of class fGARCH (GARCH
mdels fitted with the fGarch package).
}
\details{
The tests are performed using bootstrap procedures or using asymptotic
results, where applicable. Currently, two methods of generating a bootstrap
sample from the null distribution are available. The "sign" method generates
the bootstrap sample by multiplying the existing sample by -1 or 1 at random
(with equal probabilities), essentially randomizing the sign of the data,
giving a symmetric distribution. The "reflect" method reflects the sample
around zero and samples length(x) elements with replacement. In practice, it
has been shown that the "sign" method is almost always better, thus is the
default.

For numeric data, the tests can be performed around a known (parameter "mu")
or unknown centre. When the centre is known, the bootstrap method gives the
same results as a Monte Carlo simulation of the p value, for tests which are
distribution free. For unknown centre (when mu = NULL), bootstrap must be
used and the estimate of the centre used is the trimmed mean, with trim
parameter "trim". By default, the mean is taken (trim = 0).

For linear models, the tests are based on a bootstrap procedure as in
\insertCite{Allison}{symmetry} and are used to test the symmetry of the
residuals around zero.

For GARCH models (must be fitted with the 'fGarch' package), the tests are also
based on bootstrap and test for symmetry of the residuals around zero. An
approximation of the bootstrap procedure is available where the residuals are
treated as iid data, which is much faster and has been shown to give similar
results to the default bootstrap procedure (described in
\insertCite{Klar2012}{symmetry}).

For a comparison of the performance of various tests of symmetry around an
unknown centre, see \insertCite{UNKcentre}{symmetry}).
}
\examples{
set.seed(1)

# IID samples
x <- rnorm(50)
symmetry_test(x, "MOI", bootstrap = FALSE, k = 3, mu = 0)
symmetry_test(x, "MOI", bootstrap = TRUE, k = 3, mu = 0)
symmetry_test(x, "MOI", bootstrap = TRUE, k = 3)
x <- rsl(50, alpha = 1.5)
symmetry_test(x, "MOI", bootstrap = FALSE, k = 3, mu = 0)
symmetry_test(x, "MOI", bootstrap = TRUE, k = 3, mu = 0)
symmetry_test(x, "MOI", bootstrap = TRUE, k = 3)

# Linear models
lin_model <- lm(dist ~ speed, cars)
symmetry_test(lin_model, "B1")

# Garch models
library(fGarch)
specskew19 = fGarch::garchSpec(model = list(omega = 0.1,
                                    alpha = 0.3,
                                    beta = 0.3,
                                    skew = 1.9),
                                    cond.dist = "snorm")

x <- fGarch::garchSim(specskew19, n = 500)
g <- fGarch::garchFit(~garch(1,1), x, cond.dist = "QMLE",
              include.mean = FALSE, trace = FALSE)
\donttest{symmetry_test(g, "CH", B=400, burn = 100)} # slower
symmetry_test(g, "CH", B=400, burn = 100, approximate = TRUE)

}
\references{
\insertAllCited{}
}
