% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gridsearch_survdnn.R
\name{gridsearch_survdnn}
\alias{gridsearch_survdnn}
\title{Grid Search for survdnn Hyperparameters}
\usage{
gridsearch_survdnn(
  formula,
  train,
  valid,
  times,
  metrics = c("cindex", "ibs"),
  param_grid,
  .seed = 42,
  .device = c("auto", "cpu", "cuda")
)
}
\arguments{
\item{formula}{A survival formula (e.g., `Surv(time, status) ~ .`)}

\item{train}{Training dataset}

\item{valid}{Validation dataset}

\item{times}{Evaluation time points (numeric vector)}

\item{metrics}{Evaluation metrics (character vector): any of "cindex", "ibs", "brier"}

\item{param_grid}{A named list of hyperparameters (`hidden`, `lr`, `activation`, `epochs`, `loss`)}

\item{.seed}{Optional random seed for reproducibility}

\item{.device}{Character string indicating the computation device used when fitting models for each hyperparameter configuration. One of `"auto"`, `"cpu"`, or `"cuda"`. `"auto"` uses CUDA if available, otherwise falls back to CPU.}
}
\value{
A tibble with configurations and their validation metrics
}
\description{
Performs grid search over user-specified hyperparameters and evaluates performance on a validation set.
}
\examples{
\donttest{
library(survdnn)
library(survival)
set.seed(123)
# Simulate small dataset
n <- 300
x1 <- rnorm(n); x2 <- rbinom(n, 1, 0.5)
time <- rexp(n, rate = 0.1)
status <- rbinom(n, 1, 0.7)
df <- data.frame(time, status, x1, x2)
# Split into training and validation
idx <- sample(seq_len(n), 0.7 * n)
train <- df[idx, ]
valid <- df[-idx, ]
# Define formula and param grid
formula <- Surv(time, status) ~ x1 + x2
param_grid <- list(
  hidden     = list(c(16, 8), c(32, 16)),
  lr         = c(1e-3),
  activation = c("relu"),
  epochs     = c(100),
  loss       = c("cox", "coxtime")
)
# Run grid search
results <- gridsearch_survdnn(
  formula = formula,
  train   = train,
  valid   = valid,
  times   = c(10, 20, 30),
  metrics = c("cindex", "ibs"),
  param_grid = param_grid
)
# View summary
dplyr::group_by(results, hidden, lr, activation, epochs, loss, metric) |>
  dplyr::summarise(mean = mean(value, na.rm = TRUE), .groups = "drop")
}
}
