/**
 * [Grid.java]
 * 
 * Density grid class for DStream.
 * 
 * @author Yunsu Kim
 * Data Management and Data Exploration Group, RWTH Aachen University
 */
package moa.clusterers.dstream;

import java.util.Arrays;

import weka.core.Instance;


public class Grid {
	
	protected static final long TIME_NULL = -1;
	protected static final int POSITION_NULL = -1;
	protected static final double NO_CLASS = -1;
	protected enum Status {SPORADIC, NORMAL};
	protected enum Level {DENSE, TRANSITIONAL, SPARSE};
	
	protected int[] position;
	protected double len;
	
	protected long tg;
	protected long tm;
	protected double D;
	protected double label;
	protected Status status;
	protected Level level_saved;
	
	protected static double lambda;
	protected static int d;
	protected static int N;
	protected static double Cm, Cl;
	protected static double Dm, Dl;
	protected static double beta;
	
	protected static boolean staticsSet = false;
	
	/**
	 * Grid Constructor.
	 * 
	 * @param tc - current time
	 * @param classLabel
	 */
	public Grid(int[] pos, double len, long tc) {
		if (!staticsSet) {
			throw new RuntimeException("Grid: setStatics() must be called before instantiating");
		}
		
		if (pos.length != d) {
			throw new RuntimeException("Grid: given gridIndex has different #dimensions with the static setting");
		}
		
		position = Arrays.copyOf(pos, d);
		this.len = len;
		
		tg = tc;
		tm = TIME_NULL;
		D = 0;
		label = NO_CLASS;
		status = Status.NORMAL;
	}
	
	public Grid(int[] pos, double len, long tc, long tm) {
		this(pos, len, tc);
		this.tm = tm;
	}
	
	
	/* Static setters: must be called before instantiating */

	public static void setStatics(double lambda, int N, int d,
								  double Cm, double Cl) {
		Grid.lambda = lambda;
		Grid.d = d;
		Grid.N = N;
		
		Grid.Cl = Cl;
		Grid.Dm = Cm / N / (1 - lambda);
		Grid.Dl = Cl / N / (1 - lambda);
		
		Grid.staticsSet = true;
	}
	
	
	/* Updates */
	
	/**
	 * Map a new data record to this grid.
	 * 
	 * @param tn - the time new record comes in
	 */
	public void mapNewRecord(long tn) {
		updateDensity(tn);
		D++;
	}
	
	/**
	 * Update density to the current time, without a new record.
	 * 
	 * @param tc - current time
	 */
	public void updateDensity(long tc) {
		double tl = tg;
		
		D = Math.pow(lambda, tc-tl) * D;
		tg = tc;
	}
	
	public void updateStatus(long t) {
		if (isSparseGrid()) {
			if (D < densityThresholdFunction(t) && t >= (1 + beta) * tm) {
				status = Status.SPORADIC;
			} else {
				status = Status.NORMAL;
			}
		} else {
			status = Status.NORMAL;
		}
	}

	protected double densityThresholdFunction(long t) {
		return Cl * (1 - Math.pow(lambda, t - tg + 1)) / (N * (1 - lambda));
	}
	
	public void setClassLabel(double label) {
		this.label = label;
	}
	
	public void setNoClass() {
		this.label = NO_CLASS;
	}
	
	public void saveLevel() {
		if (isDenseGrid()) {
			this.level_saved = Level.DENSE;
		} else if (isTransitionalGrid()) {
			this.level_saved = Level.TRANSITIONAL;
		} else {
			this.level_saved = Level.SPARSE;
		}
	}
	
	
	/* Checking functions */
	
	public boolean isDenseGrid() {
		return D >= Dm;
	}
	
	public boolean isTransitionalGrid() {
		return D >= Dl && D <= Dm;
	}
	
	public boolean isSparseGrid() {
		return D <= Dl;
	}
	
	public Level currentLevel() {
		if (isDenseGrid()) {
			return Level.DENSE;
		} else if (isTransitionalGrid()) {
			return Level.TRANSITIONAL;
		} else {
			return Level.SPARSE;
		} 
	}
		
	public boolean isMarkedAsSporadic() {
		return status == Status.SPORADIC;
	}
	
	public boolean isNeighboringGridOf(Grid g2) {
		Grid g1 = this;
		int k = POSITION_NULL;
		
		for (int i = 0; i < d; i++) {
			if (Math.abs(g1.getPositionOfDim(i) - g2.getPositionOfDim(i)) == 1) {
				if (k == POSITION_NULL) {
					k = i;
				} else {	// More than one dimension is adjacent
					return false;
				}
			}
		}
		
		if (k == POSITION_NULL) {	// No adjacents
			return false;
		}
		
		return true;
	}
	
	public int getNeighboringDimensionOf(Grid g2) {
		Grid g1 = this;
		int k = POSITION_NULL;
		
		for (int i = 0; i < d; i++) {
			if (Math.abs(g1.getPositionOfDim(i) - g2.getPositionOfDim(i)) == 1) {
				if (k == POSITION_NULL) {
					k = i;
				} else {
					return POSITION_NULL;
				}
			}
		}
		
		return k;
	}
	
	public boolean isInsideGridIn(GridCluster G) {
		if (!G.containsGrid(this)) {
			throw new RuntimeException("Grid.isInsideGridIn(): given grid g is not a member of this group");
		}
		
		boolean[] hasNeighbor = new boolean[d];
		boolean[] allTrue = new boolean[d];
		for (int i = 0; i < d; i++) {
			hasNeighbor[i] = false;
			allTrue[i] = true;
		}
		
		for (Grid g2 : G.getGrids()) {
			int nDim = this.getNeighboringDimensionOf(g2);
			if (nDim >= 0) {
				hasNeighbor[nDim] = true;
			}
		}
		
		return Arrays.equals(hasNeighbor, allTrue);
	}
	
	public boolean isOutsideGridIn(GridCluster G) {
		return !isInsideGridIn(G);
	}
	
	public boolean includes(Instance inst) {
		if (inst.numAttributes() != d) {
			throw new RuntimeException("Grid.includes(): given instance has different dimensions with the grid");
		}
		
		for (int i = 0; i < d; i++) {
			double value = inst.value(i);
			int j = position[i];
			if (len * j < value || len * (j + 1) > value) {
				return false;
			}
		}
		
		return true;
	}
	
	
	/* Getters */
	
	public int[] getPosition() {
		return position;
	}
	
	public int getPositionOfDim(int i) {
		return position[i];
	}
	
	public long getLastDensityUpdateTime() {
		return tg;
	}
	
	public long getLastRemovedTime() {
		return tm;
	}
	
	public double getLastGridDensity() {
		return D;
	}
	
	public double getLabel() {
		return label;
	}
	
	public Status getStatus() {
		return status;
	}
	
	public Level loadSavedLevel() {
		return level_saved;
	}
}
