% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/subscreencalc.R
\name{subscreencalc}
\alias{subscreencalc}
\title{(i) Calculation of the results for the subgroups}
\usage{
subscreencalc(data, eval_function, endpoints, treat = "trtp",
  subjectid = "subjid", factors = NULL, min_comb = 1, max_comb = 3,
  nkernel = 1, par_functions = "", verbose = T)
}
\arguments{
\item{data}{dataframe with study data}

\item{eval_function}{name of the function for data analysis}

\item{endpoints}{vector containing the names of the endpoint variables in data}

\item{treat}{name of variable in data that contains the treatment identfier, defaults to trtp}

\item{subjectid}{name of variable in data that contains the subject identifier, defaults to subjid}

\item{factors}{vector containg the names of variables that define the subgroups, defaults to NULL. If set to NULL, all variables in data are used that are not included in subjectid, treat, and endpoints}

\item{min_comb}{minimum number of factor combination levels to define subgroups, defaults to 1}

\item{max_comb}{maximum number of factor combination levels to define subgruops, defaults to 3}

\item{nkernel}{number of kernels for parallelization (defaults to 1)}

\item{par_functions}{vector of names of functions used in eval_function to be exported to cluster (needed only if nkernel > 1)}

\item{verbose}{switch on/off output of computational information}
}
\value{
an object of type SubScreenResult of the form
list(sge=H,
     max_comb=max_comb,
     min_comb=min_comb,
     subjectid=subjectid,
     endpoints=endpoints,
     treat=treat,
     factors=factors,
     results_total=eval_function(cbind(F,T)))
}
\description{
This function systematically calculates the defined outcome for every combination of subgroups
up to the given level (max_comb), i.e. the number of maximum combinations of subgroup defining factors.
If, e.g., in a study sex, age group (<=60, >60), BMI group (<=25, >25) are of interest, subgroups of level 2
would be, e.g, male subjects with BMI>25 or young females, while subgroups of level 3 would be any combination
of all three variables.
}
\details{
The evaluation function (eval_function) has to defined by the user. The result needs to be a vector of numerical values,
e.g., outcome variable(s) and number of observations/subjects. The input of eval_function is a data frame with the same
structure as the input data frame (data) used in the subsreencalc call. See example below.
Potential errors occurring due to small subgroups should be caught and handled within eval_function.
As the eval_function will be called with every subgroup it may happen that there is only one observation or
only one treatment arm or only observations with missing data going into the eval_function. There should always be valid
result vector be returned (NAs allowed) and no error causing program abort.
For a better display the results may be cut-off to a reasonable range. For example: If my endpoint is a hazard ratio
that is expected to be between 0.5 and 2 I would set all values smaller than 0.01 to 0.01 and values above 100 to 100.
}
\examples{
# get the pbc data from the survival package
require(survival)
data(pbc, package="survival")

# generate categorical versions of some of the baseline covariates
pbc$ageg[!is.na(pbc$age)]        <-
   ifelse(pbc$age[!is.na(pbc$age)]          <= median(pbc$age,     na.rm=TRUE), "Low", "High")
pbc$albuming[!is.na(pbc$albumin)]<-
   ifelse(pbc$albumin[!is.na(pbc$albumin)]  <= median(pbc$albumin, na.rm=TRUE), "Low", "High")
pbc$phosg[!is.na(pbc$alk.phos)]  <-
   ifelse(pbc$alk.phos[!is.na(pbc$alk.phos)]<= median(pbc$alk.phos,na.rm=TRUE), "Low", "High")
pbc$astg[!is.na(pbc$ast)]        <-
   ifelse(pbc$ast[!is.na(pbc$ast)]          <= median(pbc$ast,     na.rm=TRUE), "Low", "High")
pbc$bilig[!is.na(pbc$bili)]      <-
   ifelse(pbc$bili[!is.na(pbc$bili)]        <= median(pbc$bili,    na.rm=TRUE), "Low", "High")
pbc$cholg[!is.na(pbc$chol)]      <-
   ifelse(pbc$chol[!is.na(pbc$chol)]        <= median(pbc$chol,    na.rm=TRUE), "Low", "High")
pbc$copperg[!is.na(pbc$copper)]  <-
   ifelse(pbc$copper[!is.na(pbc$copper)]    <= median(pbc$copper,  na.rm=TRUE), "Low", "High")

pbc$ageg[is.na(pbc$age)]         <- "No Data"
pbc$albuming[is.na(pbc$albumin)] <- "No Data"
pbc$phosg[is.na(pbc$alk.phos)]   <- "No Data"
pbc$astg[is.na(pbc$ast)]         <- "No Data"
pbc$bilig[is.na(pbc$bili)]       <- "No Data"
pbc$cholg[is.na(pbc$chol)]       <- "No Data"
pbc$copperg[is.na(pbc$copper)]   <- "No Data"
# redefine censoring variable, consider transplant/death as event
pbc$event <- pbc$status
pbc$event[pbc$event==2] <- 1

pbcdat <- pbc[!is.na(pbc$trt), ]

# define function the eval_function()
# Attention: The eval_function ALWAYS needs to return a dataframe with one row.
#            Include exception handling, like if(N1>0 && N2>0) hr <- exp(coxph(...) )
#            to avoid program abort due to errors

hazardratio <- function(x) {
   N1 <- sum(x$trt==1)
   N2 <- sum(x$trt==2)

   hr <- tryCatch(exp(coxph(Surv(time, event) ~ trt, data=x)$coefficients[[1]]),
                 warning=function(w) {NA})

   data.frame(hr=hr, N1=N1, N2=N2)
 }

 # run subscreen

\dontrun{
results <- subscreencalc(data=pbcdat,
                     eval_function=hazardratio,
                     endpoints = c("time", "event"),
                     treat="trt",
                     subjectid = "id",
                     factors=c("ageg", "sex", "bilig", "cholg", "copperg", "astg",
                               "albuming", "phosg"))

# visualize the results of the subgroup screening with a Shiny app
subscreenshow(results)}
}
\keyword{analysis}
\keyword{subgroup}
