% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/STR.R
\name{STR}
\alias{STR}
\title{Automatic STR decomposition}
\usage{
STR(data, predictors, confidence = NULL, robust = FALSE, lambdas = NULL,
  pattern = extractPattern(predictors), nFold = 5, reltol = 0.005,
  gapCV = 1, solver = c("Matrix", "cholesky"), nMCIter = 100,
  control = list(nnzlmax = 1e+06, nsubmax = 3e+05, tmpmax = 50000),
  trace = FALSE, iterControl = list(maxiter = 20, tol = 1e-06))
}
\arguments{
\item{data}{Time series or a vector of length \emph{\strong{L}}.}

\item{predictors}{List of predictors.\cr
According to the paradigm of this implementation, the trend, the seasonal components,
the flexible predictors and the seasonal predictors are all presented in the same
form (as predictors) and must be described in this list.\cr
Every predictor is a list of the following structures:\cr
\itemize{
\item \strong{data} -- vector of length \emph{\strong{L}} (length of input data,
  see above). For trend or for a seasonal component it is a vector of ones.
  For a flexible or a seasonal predictor it is a vector of the predictor's data.
\item \strong{times} -- vector of length \emph{\strong{L}} of times of observations.
\item \strong{seasons} -- vector of length \emph{\strong{L}}. It is a vector of ones
  for a trend or a flexible predictor. It is vector assigning seasons to every
  observation (for a seasonal component or a seasonal predictor).
  Seasons can be fractional for observations in between seasons.
\item \strong{timeKnots} -- vector of times (time knots) where knots are positioned
  (for a seasonal component or a seasonal predictor a few knots have the same time;
  every knot is represented by time and season). Usually this vector coincides with
  \strong{times} vector described above, or \strong{timeKnots} is a subset of
  \strong{times} vector.
\item \strong{seasonalStructure} -- describes seasonal topology (which can have complex
  structure) and seasonal knots.The seasonal topology is described by a list of
  segments and seasonal knots, which are positioned inside the segments, on borders of
  the segments or, when they are on on borders, they can connect two or more segments.\cr
\strong{seasonalStructure} is a list of two elements:\cr
\itemize{
\item \strong{segments} -- a list of vectors representing segments.
Each vector must contain two ordered real values which represent left and right borders
  of a segment. Segments should not intersect (inside same predictor).
\item \strong{sKnots} -- a list of real values (vectors of length one) or vectors of
  lengths two or greater (seasonal knots) defining seasons of the knots (every knot
  is represented by time and season). All real values must belong (be inside or on
  border of) segments listed in \strong{segments}. If a few values represent a single
  seasonal knot then all these values must be on borders of some segments (or a single
  segment). In this case they represent a seasonal knot which connects a few segments
  (or both sides of one segment).
}
\item \strong{lambdas} -- a vector with three values representing lambda (smoothing)
  parameters (time-time, season-season, time-season flexibility parameters) for this
  predictor.
}}

\item{confidence}{A vector of percentiles giving the coverage of confidence intervals.
It must be greater than 0 and less than 1.
If \code{NULL}, no confidence intervals are produced.}

\item{robust}{When \code{TRUE}, Robust STR decomposition is used. Default is \code{FALSE}.}

\item{lambdas}{An optional parameter.
A structure which replaces lambda parameters provided with predictors.
It is used as either a starting point for the optimisation of parameters or as the
exact model parameters.}

\item{pattern}{An optional parameter which has the same structure as \code{lambdas}
although with a different meaning. All zero values correspond to lambda
(smoothing) parameters which will not be estimated.}

\item{nFold}{An optional parameter setting the number of folds for cross validation.}

\item{reltol}{An optional parameter which is passed directly to \code{\link{optim}()}
when optimising the parameters of the model.}

\item{gapCV}{An optional parameter defining the length of the sequence of
skipped values in the cross validation procedure.}

\item{solver}{A vector with two string values. The only supported combinations are:
c("Matrix", "cholesky") (default), and c("Matrix", "qr").
The parameter is used to specify a particular library
and method to solve the minimisation problem during STR decompositon.}

\item{nMCIter}{Number of Monte Carlo iterations used to estimate confidence intervals for Robust STR decomposition.}

\item{control}{Passed directly to \code{\link{rq.fit.sfn}()} during Robust STR decomposition.}

\item{trace}{When \code{TRUE}, tracing is turned on.}

\item{iterControl}{Control parameters for some experimental features.
This should not be used by an ordinary user.}
}
\value{
A structure containing input and output data.
It is an \strong{S3} class \code{STR}, which is a list with the following components:
\itemize{
\item \strong{output} -- contains decomposed data. It is a list of three components:
\itemize{
\item \strong{predictors} -- a list of components where each component
corresponds to the input predictor. Every such component is a list containing the following:
\itemize{
\item \strong{data} -- fit/forecast for the corresponding predictor (trend, seasonal component, flexible or seasonal predictor).
\item \strong{beta} -- beta coefficients of the fit of the coresponding predictor.
\item \strong{lower} -- optional (if requested) matrix of lower bounds of confidence intervals.
\item \strong{upper} -- optional (if requested) matrix of upper bounds of confidence intervals.
}
\item \strong{random} -- a list with one component \strong{data}, which contains residuals of the model fit.
\item \strong{forecast} -- a list with two components:
\itemize{
\item \strong{data} -- fit/forecast for the model.
\item \strong{beta} -- beta coefficients of the fit.
\item \strong{lower} -- optional (if requested) matrix of lower bounds of confidence intervals.
\item \strong{upper} -- optional (if requested) matrix of upper bounds of confidence intervals.
}
}
\item \strong{input} -- input parameters and lambdas used for final calculations.
\itemize{
\item \strong{data} -- input data.
\item \strong{predictors} - input predictors.
\item \strong{lambdas} -- smoothing parameters used for final calculations (same as input lambdas for STR method).
}
\item \strong{cvMSE} -- optional cross validated (leave one out) Mean Squared Error.
\item \strong{optim.CV.MSE} or \strong{optim.CV.MAE} -- best cross validated Mean Squared Error or Mean Absolute Error (n-fold) achieved during minimisation procedure.
\item \strong{nFold} -- the input \code{nFold} parameter.
\item \strong{gapCV} -- the input \code{gapCV} parameter.
\item \strong{method} -- contains strings \code{"STR"} or \code{"RSTR"} depending on used method.
}
}
\description{
Automatically selects parameters for an STR decomposition of time series data.

If a parallel backend is registered for use before \code{STR} call,
\code{STR} will use it for n-fold cross validation computations.
}
\examples{
\donttest{

TrendSeasonalStructure <- list(segments = list(c(0,1)),
sKnots = list(c(1,0)))
WDSeasonalStructure <- list(segments = list(c(0,48), c(100,148)),
                            sKnots = c(as.list(c(1:47,101:147)), list(c(0,48,100,148))))

TrendSeasons <- rep(1, nrow(electricity))
WDSeasons <- as.vector(electricity[,"WorkingDaySeasonality"])

Data <- as.vector(electricity[,"Consumption"])
Times <- as.vector(electricity[,"Time"])
TempM <- as.vector(electricity[,"Temperature"])
TempM2 <- TempM^2

TrendTimeKnots <- seq(from = head(Times, 1), to = tail(Times, 1), length.out = 116)
SeasonTimeKnots <- seq(from = head(Times, 1), to = tail(Times, 1), length.out = 24)

TrendData <- rep(1, length(Times))
SeasonData <- rep(1, length(Times))

Trend <- list(name = "Trend",
              data = TrendData,
              times = Times,
              seasons = TrendSeasons,
              timeKnots = TrendTimeKnots,
              seasonalStructure = TrendSeasonalStructure,
              lambdas = c(1500,0,0))
WDSeason <- list(name = "Dayly seas",
                 data = SeasonData,
                 times = Times,
                 seasons = WDSeasons,
                 timeKnots = SeasonTimeKnots,
                 seasonalStructure = WDSeasonalStructure,
                 lambdas = c(0.003,0,240))
StaticTempM <- list(name = "Temp Mel",
                    data = TempM,
                    times = Times,
                    seasons = NULL,
                    timeKnots = NULL,
                    seasonalStructure = NULL,
                    lambdas = c(0,0,0))
StaticTempM2 <- list(name = "Temp Mel^2",
                     data = TempM2,
                     times = Times,
                     seasons = NULL,
                     timeKnots = NULL,
                     seasonalStructure = NULL,
                     lambdas = c(0,0,0))
Predictors <- list(Trend, WDSeason, StaticTempM, StaticTempM2)

elec.fit <- STR(data = Data,
                predictors = Predictors,
                gapCV = 48*7)

plot(elec.fit,
     xTime = as.Date("2000-01-11")+((Times-1)/48-10),
     forecastPanels = NULL)

#########################################################

n <- 70
trendSeasonalStructure <- list(segments = list(c(0,1)), sKnots = list(c(1,0)))
ns <- 5
seasonalStructure <- list(segments = list(c(0,ns)), sKnots = c(as.list(1:(ns-1)),list(c(ns,0))))
seasons <- (0:(n-1))\%\%ns + 1
trendSeasons <- rep(1, length(seasons))
times <- seq_along(seasons)
data <- seasons + times/4
set.seed(1234567890)
data <- data + rnorm(length(data), 0, 0.2)
data[20] <- data[20]+3
data[50] <- data[50]-5
plot(times, data, type = "l")
timeKnots <- times
trendData <- rep(1, n)
seasonData <- rep(1, n)
trend <- list(data = trendData, times = times, seasons = trendSeasons,
  timeKnots = timeKnots, seasonalStructure = trendSeasonalStructure, lambdas = c(1,0,0))
season <- list(data = seasonData, times = times, seasons = seasons,
  timeKnots = timeKnots, seasonalStructure = seasonalStructure, lambdas = c(1,0,1))
predictors <- list(trend, season)
rstr <- STR(data, predictors, reltol = 0.0000001, gapCV = 10,
                confidence = 0.95, nMCIter = 400, robust = TRUE)
plot(rstr)
}
}
\author{
Alexander Dokumentov
}
\references{
Dokumentov, A., and Hyndman, R.J. (2016)
STR: A Seasonal-Trend Decomposition Procedure Based on Regression
\href{https://www.monash.edu/business/econometrics-and-business-statistics/research/publications/ebs/wp13-15.pdf}{www.monash.edu/business/econometrics-and-business-statistics/research/publications/ebs/wp13-15.pdf}
}
\seealso{
\code{\link{STRmodel}} \code{\link{RSTRmodel}} \code{\link{AutoSTR}}
}

